import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Plan Execution ===

        # [Step 1] Move to the side position of the bottom drawer
        try:
            side_pos = positions.get('bottom_side_pos')
            if side_pos is None:
                raise KeyError("bottom_side_pos not found in positions.")
            obs, reward, done = move(env, task, target_pos=np.array(side_pos))
            if done:
                print("[Task] Task ended after move to side!")
                return
        except Exception as e:
            print(f"[Task] Exception during move to side: {e}")
            shutdown_environment(env)
            return

        # [Step 2] Rotate gripper to 90 degrees (for side approach)
        try:
            from scipy.spatial.transform import Rotation as R
            quat_90_z = R.from_euler('z', 90, degrees=True).as_quat()  # xyzw
            obs, reward, done = rotate(env, task, target_quat=quat_90_z)
            if done:
                print("[Task] Task ended after rotate!")
                return
        except Exception as e:
            print(f"[Task] Exception during rotate: {e}")
            shutdown_environment(env)
            return

        # [Step 3] Move to anchor position of bottom drawer
        try:
            anchor_pos = positions.get('bottom_anchor_pos')
            if anchor_pos is None:
                raise KeyError("bottom_anchor_pos not found in positions.")
            obs, reward, done = move(env, task, target_pos=np.array(anchor_pos))
            if done:
                print("[Task] Task ended after move to anchor!")
                return
        except Exception as e:
            print(f"[Task] Exception during move to anchor: {e}")
            shutdown_environment(env)
            return

        # [Step 4] Pick (grasp) the drawer handle at anchor position
        try:
            obs, reward, done = pick(env, task, target_pos=np.array(anchor_pos), approach_distance=0.10, approach_axis='z')
            if done:
                print("[Task] Task ended after pick drawer!")
                return
        except Exception as e:
            print(f"[Task] Exception during pick drawer: {e}")
            shutdown_environment(env)
            return

        # [Step 5] Pull open the drawer (assume +x direction, adjust if needed)
        try:
            pull_distance = 0.18  # meters, adjust as needed for your environment
            obs, reward, done = pull(env, task, pull_distance=pull_distance, pull_axis='x')
            if done:
                print("[Task] Task ended after pull!")
                return
        except Exception as e:
            print(f"[Task] Exception during pull drawer: {e}")
            shutdown_environment(env)
            return

        # [Step 6] Move to the rubbish position
        try:
            rubbish_pos = positions.get('rubbish')
            if rubbish_pos is None:
                raise KeyError("rubbish not found in positions.")
            obs, reward, done = move(env, task, target_pos=np.array(rubbish_pos))
            if done:
                print("[Task] Task ended after move to rubbish!")
                return
        except Exception as e:
            print(f"[Task] Exception during move to rubbish: {e}")
            shutdown_environment(env)
            return

        # [Step 7] Pick up the rubbish from the table
        try:
            obs, reward, done = pick(env, task, target_pos=np.array(rubbish_pos), approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended after pick rubbish!")
                return
        except Exception as e:
            print(f"[Task] Exception during pick rubbish: {e}")
            shutdown_environment(env)
            return

        # [Step 8] Move to the bin
        try:
            bin_pos = positions.get('bin')
            if bin_pos is None:
                raise KeyError("bin not found in positions.")
            obs, reward, done = move(env, task, target_pos=np.array(bin_pos))
            if done:
                print("[Task] Task ended after move to bin!")
                return
        except Exception as e:
            print(f"[Task] Exception during move to bin: {e}")
            shutdown_environment(env)
            return

        # [Step 9] Place the rubbish into the bin
        try:
            obs, reward, done = place(env, task, target_pos=np.array(bin_pos), approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task completed successfully! Reward:", reward)
            else:
                print("[Task] Task not completed yet (done=False).")
        except Exception as e:
            print(f"[Task] Exception during place rubbish: {e}")
            shutdown_environment(env)
            return

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()