import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use provided skills only

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Task: Open a drawer fully, pick up the rubbish, and place it in the bin.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # --- Object name mapping (from object list and environment) ---
        # We'll use the 'bottom' drawer for this task.
        # Key objects: 'rubbish', 'bin', 'bottom_anchor_pos', 'bottom_side_pos', 'bottom_joint_pos'
        # The gripper starts at 'nowhere_pos' and must move to the drawer's side, then anchor, then pull.

        # Defensive: check all required keys exist
        required_keys = [
            'rubbish', 'bin',
            'bottom_side_pos', 'bottom_anchor_pos', 'bottom_joint_pos'
        ]
        for k in required_keys:
            if k not in positions:
                raise RuntimeError(f"Required object position '{k}' not found in get_object_positions().")

        rubbish_pos = positions['rubbish']
        bin_pos = positions['bin']
        bottom_side_pos = positions['bottom_side_pos']
        bottom_anchor_pos = positions['bottom_anchor_pos']
        bottom_joint_pos = positions['bottom_joint_pos']

        # --- Step 1: Rotate gripper to 90 degrees (vertical for drawer handle) ---
        # The skill expects a quaternion. We'll use scipy to get the quaternion for 90deg about Z.
        from scipy.spatial.transform import Rotation as R
        # The initial orientation is assumed to be zero_deg (identity quaternion)
        # 90 degrees about Z axis
        target_rot = R.from_euler('z', 90, degrees=True)
        target_quat = target_rot.as_quat()  # xyzw
        # Ensure it's numpy array
        target_quat = np.array(target_quat)
        # The skill expects xyzw order
        print("[Task] Rotating gripper to 90 degrees about Z axis for drawer operation.")
        pull_distance = 0.20  # Define here for later use

        # [Frozen Code Start]
        obs, reward, done = rotate(env, task, target_quat)
        obs, reward, done = move(env, task, bottom_side_pos)
        obs, reward, done = move(env, task, bottom_anchor_pos)
        obs, reward, done = pick(env, task, bottom_anchor_pos, approach_distance=0.10, approach_axis='z')
        obs, reward, done = pull(env, task, pull_distance, pull_axis='x')
        # [Frozen Code End]

        if done:
            print("[Task] Task ended during drawer opening steps!")
            return

        # --- Step 6: Pick up the rubbish from inside the drawer ---
        # Move above the rubbish (if needed), then pick
        print("[Task] Moving gripper above rubbish to pick it up.")
        # Approach from above: offset in z
        rubbish_above = np.array(rubbish_pos) + np.array([0, 0, 0.10])
        obs, reward, done = move(env, task, rubbish_above)
        if done:
            print("[Task] Task ended during move above rubbish!")
            return
        print("[Task] Picking up the rubbish.")
        obs, reward, done = pick(env, task, rubbish_pos, approach_distance=0.10, approach_axis='z')
        if done:
            print("[Task] Task ended during pick (rubbish)!")
            return

        # --- Step 7: Place the rubbish in the bin ---
        print("[Task] Moving gripper above bin to place rubbish.")
        bin_above = np.array(bin_pos) + np.array([0, 0, 0.10])
        obs, reward, done = move(env, task, bin_above)
        if done:
            print("[Task] Task ended during move above bin!")
            return
        print("[Task] Placing the rubbish in the bin.")
        obs, reward, done = place(env, task, bin_pos, approach_distance=0.10, approach_axis='z')
        if done:
            print("[Task] Task completed successfully! Reward:", reward)
        else:
            print("[Task] Task not completed yet (done=False).")

    except Exception as e:
        print(f"[Task] Exception occurred: {e}")
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()