import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === PLAN EXECUTION ===
        # Goal: Slide open any available drawer to access its contents, then chuck away any rubbish remaining on the table.

        # 1. Choose a drawer to open (bottom, middle, or top). We'll use 'bottom' drawer.
        # 2. Find the relevant positions for the gripper to approach, anchor, and pull the drawer.
        # 3. After opening, pick up 'rubbish' from the table and place it in the bin.

        # --- Drawer Opening Sequence ---
        # We'll use the following convention for object_positions:
        #   - 'bottom_side_pos': side position for bottom drawer
        #   - 'bottom_anchor_pos': anchor position for bottom drawer
        #   - 'bottom_joint_pos': joint (handle) position for bottom drawer

        # Step 1: Rotate gripper to 90 degrees (vertical for grasping handle)
        try:
            # Get current gripper pose
            obs = task.get_observation()
            start_quat = obs.gripper_pose[3:7]
            # Target orientation: 90 deg about z axis (assuming 'z' is up)
            # We'll use scipy to get the quaternion for 90 deg about z
            from scipy.spatial.transform import Rotation as R
            target_rot = R.from_euler('z', 90, degrees=True)
            target_quat = target_rot.as_quat()  # xyzw
            obs, reward, done = rotate(env, task, target_quat)
            if done:
                print("[Task] Task ended during rotate!")
                return
        except Exception as e:
            print(f"[ERROR] rotate failed: {e}")
            return

        # Step 2: Move to side position of bottom drawer
        try:
            side_pos = positions.get('bottom_side_pos', None)
            if side_pos is None:
                raise ValueError("bottom_side_pos not found in object positions.")
            obs, reward, done = move(env, task, side_pos)
            if done:
                print("[Task] Task ended during move to side position!")
                return
        except Exception as e:
            print(f"[ERROR] move to side position failed: {e}")
            return

        # Step 3: Move to anchor position of bottom drawer (for grasping handle)
        try:
            anchor_pos = positions.get('bottom_anchor_pos', None)
            if anchor_pos is None:
                raise ValueError("bottom_anchor_pos not found in object positions.")
            obs, reward, done = move(env, task, anchor_pos)
            if done:
                print("[Task] Task ended during move to anchor position!")
                return
        except Exception as e:
            print(f"[ERROR] move to anchor position failed: {e}")
            return

        # Step 4: Pick (grasp) the drawer handle at anchor position
        try:
            obs, reward, done = pick(env, task, anchor_pos, approach_distance=0.10, approach_axis='z')
            if done:
                print("[Task] Task ended during pick drawer handle!")
                return
        except Exception as e:
            print(f"[ERROR] pick drawer handle failed: {e}")
            return

        # Step 5: Pull the drawer open (along x axis, positive direction)
        try:
            # Estimate a reasonable pull distance (e.g., 0.15m)
            pull_distance = 0.15
            obs, reward, done = pull(env, task, pull_distance, pull_axis='x')
            if done:
                print("[Task] Task ended during pull!")
                return
        except Exception as e:
            print(f"[ERROR] pull failed: {e}")
            return

        # --- Rubbish Disposal Sequence ---
        # Step 6: Pick up 'rubbish' from the table
        try:
            rubbish_pos = positions.get('rubbish', None)
            if rubbish_pos is None:
                raise ValueError("rubbish position not found in object positions.")
            obs, reward, done = pick(env, task, rubbish_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended during pick rubbish!")
                return
        except Exception as e:
            print(f"[ERROR] pick rubbish failed: {e}")
            return

        # Step 7: Place rubbish in the bin
        try:
            bin_pos = positions.get('bin', None)
            if bin_pos is None:
                raise ValueError("bin position not found in object positions.")
            obs, reward, done = place(env, task, bin_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task completed successfully! Reward:", reward)
            else:
                print("[Task] Task not completed yet (done=False).")
        except Exception as e:
            print(f"[ERROR] place rubbish failed: {e}")
            return

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()