import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use provided skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === PLAN EXECUTION ===
        # Goal: Open one of the drawers in the cabinet, then put the rubbish in the bin.
        # We'll open the "bottom" drawer (as an example).
        # We'll assume "item3" is the rubbish to be placed in the bin.

        # --- Step 1: Move gripper to the side position of the bottom drawer ---
        try:
            bottom_side_pos = positions.get('bottom_side_pos', None)
            if bottom_side_pos is None:
                raise KeyError("bottom_side_pos not found in object positions.")
            print("[Task] Moving to bottom_side_pos:", bottom_side_pos)
            [Frozen Code Start]
            obs, reward, done = move(env, task, target_pos=bottom_side_pos)
            [Frozen Code End]
            if done:
                print("[Task] Task ended after move to bottom_side_pos!")
                return
        except Exception as e:
            print(f"[Error] Failed at move to bottom_side_pos: {e}")
            return

        # --- Step 2: Rotate gripper to 90 degrees (for drawer grasp) ---
        try:
            # We'll use a quaternion for 90 deg rotation about z axis.
            # Assuming initial orientation is zero_deg (no rotation), and 90 deg is about z.
            # We'll get the current position and orientation, and rotate about z.
            obs = task.get_observation()
            current_quat = obs.gripper_pose[3:7]
            # 90 deg about z axis in quaternion (xyzw): [0, 0, sqrt(0.5), sqrt(0.5)]
            target_quat = np.array([0, 0, np.sqrt(0.5), np.sqrt(0.5)])
            print("[Task] Rotating gripper to 90 degrees (about z axis).")
            [Frozen Code Start]
            obs, reward, done = rotate(env, task, target_quat=target_quat)
            [Frozen Code End]
            if done:
                print("[Task] Task ended after rotate!")
                return
        except Exception as e:
            print(f"[Error] Failed at rotate: {e}")
            return

        # --- Step 3: Move gripper to anchor position of the bottom drawer (for grasping handle) ---
        try:
            bottom_anchor_pos = positions.get('bottom_anchor_pos', None)
            if bottom_anchor_pos is None:
                raise KeyError("bottom_anchor_pos not found in object positions.")
            print("[Task] Moving to bottom_anchor_pos:", bottom_anchor_pos)
            [Frozen Code Start]
            obs, reward, done = move(env, task, target_pos=bottom_anchor_pos)
            [Frozen Code End]
            if done:
                print("[Task] Task ended after move to bottom_anchor_pos!")
                return
        except Exception as e:
            print(f"[Error] Failed at move to bottom_anchor_pos: {e}")
            return

        # --- Step 4: Pull the drawer open ---
        try:
            # We'll pull along the x axis (assuming drawer opens along x).
            # Let's use a reasonable pull distance (e.g., 0.15 meters).
            pull_distance = 0.15
            print("[Task] Pulling the drawer open along x axis.")
            [Frozen Code Start]
            obs, reward, done = pull(env, task, pull_distance=pull_distance, pull_axis='x')
            [Frozen Code End]
            if done:
                print("[Task] Task ended after pull!")
                return
        except Exception as e:
            print(f"[Error] Failed at pull: {e}")
            return

        # --- Step 5: Move to the rubbish (item3) on the table ---
        try:
            item3_pos = positions.get('item3', None)
            if item3_pos is None:
                raise KeyError("item3 not found in object positions.")
            print("[Task] Moving to item3 (rubbish) position:", item3_pos)
            [Frozen Code Start]
            obs, reward, done = move(env, task, target_pos=item3_pos)
            [Frozen Code End]
            if done:
                print("[Task] Task ended after move to item3!")
                return
        except Exception as e:
            print(f"[Error] Failed at move to item3: {e}")
            return

        # --- Step 6: Pick up the rubbish (item3) ---
        try:
            print("[Task] Picking up item3 (rubbish).")
            [Frozen Code Start]
            obs, reward, done = pick(env, task, target_pos=item3_pos)
            [Frozen Code End]
            if done:
                print("[Task] Task ended after pick!")
                return
        except Exception as e:
            print(f"[Error] Failed at pick: {e}")
            return

        # --- Step 7: Move to the bin ---
        try:
            bin_pos = positions.get('bin', None)
            if bin_pos is None:
                raise KeyError("bin not found in object positions.")
            print("[Task] Moving to bin position:", bin_pos)
            obs, reward, done = move(env, task, target_pos=bin_pos)
            if done:
                print("[Task] Task ended after move to bin!")
                return
        except Exception as e:
            print(f"[Error] Failed at move to bin: {e}")
            return

        # --- Step 8: Place the rubbish in the bin ---
        try:
            print("[Task] Placing item3 (rubbish) in the bin.")
            obs, reward, done = place(env, task, target_pos=bin_pos)
            if done:
                print("[Task] Task completed successfully! Reward:", reward)
            else:
                print("[Task] Task not completed yet (done=False).")
        except Exception as e:
            print(f"[Error] Failed at place: {e}")
            return

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()