import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use provided skills only

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === PLAN EXECUTION ===
        # Goal: Open one of the drawers in the cabinet, then put the rubbish in the bin.
        # We'll open the "bottom" drawer (as it's likely unlocked and reachable).
        # We'll move the gripper to the side of the bottom drawer, rotate, move to anchor, pull to open,
        # then pick up the rubbish and place it in the bin.

        # --- Step 1: Move to the side position of the bottom drawer ---
        try:
            bottom_side_pos = positions.get('bottom_side_pos')
            if bottom_side_pos is None:
                raise KeyError("bottom_side_pos not found in object positions.")
            print("[Task] Moving to bottom_side_pos:", bottom_side_pos)
            [Frozen Code Start]
            obs, reward, done = move(env, task, target_pos=bottom_side_pos)
            [Frozen Code End]
            if done:
                print("[Task] Task ended after move to bottom_side_pos!")
                return
        except Exception as e:
            print(f"[Error] Failed at move to bottom_side_pos: {e}")
            return

        # --- Step 2: Rotate gripper to 90 degrees (for drawer handle alignment) ---
        try:
            # The skill expects a quaternion; let's get the quaternion for 90 deg about z axis.
            from scipy.spatial.transform import Rotation as R
            target_rot = R.from_euler('z', 90, degrees=True)
            target_quat = target_rot.as_quat()  # xyzw
            print("[Task] Rotating gripper to 90 degrees (about z axis):", target_quat)
            [Frozen Code Start]
            obs, reward, done = rotate(env, task, target_quat=target_quat)
            [Frozen Code End]
            if done:
                print("[Task] Task ended after rotate!")
                return
        except Exception as e:
            print(f"[Error] Failed at rotate: {e}")
            return

        # --- Step 3: Move to anchor position of the bottom drawer (handle) ---
        try:
            bottom_anchor_pos = positions.get('bottom_anchor_pos')
            if bottom_anchor_pos is None:
                raise KeyError("bottom_anchor_pos not found in object positions.")
            print("[Task] Moving to bottom_anchor_pos:", bottom_anchor_pos)
            [Frozen Code Start]
            obs, reward, done = move(env, task, target_pos=bottom_anchor_pos)
            [Frozen Code End]
            if done:
                print("[Task] Task ended after move to bottom_anchor_pos!")
                return
        except Exception as e:
            print(f"[Error] Failed at move to bottom_anchor_pos: {e}")
            return

        # --- Step 4: Pull to open the drawer ---
        try:
            # Pull along the x-axis (assume drawer opens along x, positive direction)
            # Use a reasonable pull distance (e.g., 0.15 meters)
            pull_distance = 0.15
            print(f"[Task] Pulling drawer by {pull_distance} meters along x-axis.")
            [Frozen Code Start]
            obs, reward, done = pull(env, task, pull_distance=pull_distance, pull_axis='x')
            [Frozen Code End]
            if done:
                print("[Task] Task ended after pull!")
                return
        except Exception as e:
            print(f"[Error] Failed at pull: {e}")
            return

        # --- Step 5: Move to the rubbish (item3) on the table ---
        try:
            item3_pos = positions.get('item3')
            if item3_pos is None:
                raise KeyError("item3 not found in object positions.")
            print("[Task] Moving to item3 (rubbish) position:", item3_pos)
            [Frozen Code Start]
            obs, reward, done = move(env, task, target_pos=item3_pos)
            [Frozen Code End]
            if done:
                print("[Task] Task ended after move to item3!")
                return
        except Exception as e:
            print(f"[Error] Failed at move to item3: {e}")
            return

        # --- Step 6: Pick the rubbish (item3) ---
        try:
            print("[Task] Picking up item3 (rubbish).")
            [Frozen Code Start]
            obs, reward, done = pick(env, task, target_pos=item3_pos)
            [Frozen Code End]
            if done:
                print("[Task] Task ended after pick!")
                return
        except Exception as e:
            print(f"[Error] Failed at pick: {e}")
            return

        # --- Step 7: Move to the bin ---
        try:
            bin_pos = positions.get('bin')
            if bin_pos is None:
                raise KeyError("bin not found in object positions.")
            print("[Task] Moving to bin position:", bin_pos)
            obs, reward, done = move(env, task, target_pos=bin_pos)
            if done:
                print("[Task] Task ended after move to bin!")
                return
        except Exception as e:
            print(f"[Error] Failed at move to bin: {e}")
            return

        # --- Step 8: Place the rubbish in the bin ---
        try:
            print("[Task] Placing item3 (rubbish) in the bin.")
            obs, reward, done = place(env, task, target_pos=bin_pos)
            if done:
                print("[Task] Task completed successfully! Reward:", reward)
            else:
                print("[Task] Task not completed yet (done=False).")
        except Exception as e:
            print(f"[Error] Failed at place: {e}")
            return

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()