import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === PLAN EXECUTION ===
        # Goal: Pull open any drawer that is not locked, then drop the rubbish into the bin.
        # Plan (from specification):
        # 1. rotate
        # 2. move
        # 3. move
        # 4. pick
        # 5. pull
        # 6. pick
        # 7. place

        # --- Step 1: Rotate gripper to 90 degrees (if required for drawer manipulation) ---
        try:
            # For the drawer, we typically need the gripper at 90 degrees (vertical).
            # Let's get the current gripper pose and rotate to 90 deg about z.
            obs = task.get_observation()
            start_quat = obs.gripper_pose[3:7]
            # 90 deg about z axis in quaternion (xyzw)
            from scipy.spatial.transform import Rotation as R
            target_rot = R.from_euler('z', 90, degrees=True)
            target_quat = target_rot.as_quat()  # xyzw
            print("[Task] Rotating gripper to 90 degrees about z axis.")
            [Frozen Code Start]
            obs, reward, done = rotate(
                env,
                task,
                target_quat=target_quat,
                max_steps=100,
                threshold=0.05,
                timeout=10.0
            )
            [Frozen Code End]
            if done:
                print("[Task] Task ended after rotate! Exiting.")
                return
        except Exception as e:
            print(f"[Error] Failed to rotate gripper: {e}")
            shutdown_environment(env)
            return

        # --- Step 2: Move to the side position of the bottom drawer ---
        try:
            side_pos = positions.get('bottom_side_pos', None)
            if side_pos is None:
                raise KeyError("Could not find 'bottom_side_pos' in object_positions.")
            print("[Task] Moving to bottom drawer side position:", side_pos)
            [Frozen Code Start]
            obs, reward, done = move(
                env,
                task,
                target_pos=np.array(side_pos)
            [Frozen Code End]
            if done:
                print("[Task] Task ended after move to side position! Exiting.")
                return
        except Exception as e:
            print(f"[Error] Failed to move to bottom drawer side position: {e}")
            shutdown_environment(env)
            return

        # --- Step 3: Move to the anchor (handle) position of the bottom drawer ---
        try:
            anchor_pos = positions.get('bottom_anchor_pos', None)
            if anchor_pos is None:
                raise KeyError("Could not find 'bottom_anchor_pos' in object_positions.")
            print("[Task] Moving to bottom drawer anchor (handle) position:", anchor_pos)
            [Frozen Code Start]
            obs, reward, done = move(
                env,
                task,
                target_pos=np.array(anchor_pos)
            [Frozen Code End]
            if done:
                print("[Task] Task ended after move to anchor position! Exiting.")
                return
        except Exception as e:
            print(f"[Error] Failed to move to bottom drawer anchor position: {e}")
            shutdown_environment(env)
            return

        # --- Step 4: Pick the drawer handle (simulate grasp) ---
        try:
            # For the drawer, we simulate a pick at the anchor position.
            print("[Task] Picking (grasping) the bottom drawer handle at:", anchor_pos)
            obs, reward, done = pick(
                env,
                task,
                target_pos=np.array(anchor_pos),
                approach_distance=0.10,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Task] Task ended after pick (drawer handle)! Exiting.")
                return
        except Exception as e:
            print(f"[Error] Failed to pick (grasp) the bottom drawer handle: {e}")
            shutdown_environment(env)
            return

        # --- Step 5: Pull open the bottom drawer ---
        try:
            # Pull along +x axis by 0.15m (typical drawer pull)
            pull_distance = 0.15
            pull_axis = 'x'
            print(f"[Task] Pulling the bottom drawer along {pull_axis} by {pull_distance}m.")
            obs, reward, done = pull(
                env,
                task,
                pull_distance=pull_distance,
                pull_axis=pull_axis,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            if done:
                print("[Task] Task ended after pull! Exiting.")
                return
        except Exception as e:
            print(f"[Error] Failed to pull open the drawer: {e}")
            shutdown_environment(env)
            return

        # --- Step 6: Pick up the rubbish from the table ---
        try:
            # The object list says 'item3' is rubbish.
            rubbish_pos = positions.get('item3', None)
            if rubbish_pos is None:
                raise KeyError("Could not find 'item3' (rubbish) position in object_positions.")
            print("[Task] Picking up 'item3' (rubbish) at:", rubbish_pos)
            obs, reward, done = pick(
                env,
                task,
                target_pos=np.array(rubbish_pos),
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Task] Task ended after pick (rubbish)! Exiting.")
                return
        except Exception as e:
            print(f"[Error] Failed to pick up 'item3' (rubbish): {e}")
            shutdown_environment(env)
            return

        # --- Step 7: Place the rubbish into the bin ---
        try:
            bin_pos = positions.get('bin', None)
            if bin_pos is None:
                raise KeyError("Could not find 'bin' position in object_positions.")
            print("[Task] Placing 'item3' (rubbish) into the bin at:", bin_pos)
            obs, reward, done = place(
                env,
                task,
                target_pos=np.array(bin_pos),
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Task] Task completed successfully! Reward:", reward)
            else:
                print("[Task] Task not completed yet (done=False).")
        except Exception as e:
            print(f"[Error] Failed to place 'item3' (rubbish) into the bin: {e}")
            shutdown_environment(env)
            return

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()