import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only the provided skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Task: Unlock a cabinet by opening an accessible drawer, then throw away the trash while leaving any other objects alone.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Object names from object list and environment:
        # Drawers: 'bottom', 'middle', 'top'
        # Drawer positions: e.g., 'bottom_anchor_pos', 'bottom_side_pos', etc.
        # Items: 'item1', 'item2', 'item3'
        # Bin: 'bin'
        # Gripper: assumed to be controlled via skills

        # --- Step 1: Move to the side position of the bottom drawer (assume bottom is accessible and not locked) ---
        try:
            bottom_side_pos = positions.get('bottom_side_pos', None)
            if bottom_side_pos is None:
                raise KeyError("bottom_side_pos not found in object positions.")
            print("[Task] Moving to bottom drawer side position:", bottom_side_pos)
            obs, reward, done = move(
                env,
                task,
                target_pos=np.array(bottom_side_pos),
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            if done:
                print("[Task] Task ended after move to bottom drawer side position!")
                return
        except Exception as e:
            print(f"[Error] Failed to move to bottom drawer side position: {e}")
            shutdown_environment(env)
            return

        # --- Step 2: Rotate gripper to 90 degrees (for drawer manipulation) ---
        try:
            # Assume 90 deg is [x, y, z, w] quaternion; get from positions or define
            # If not available, use a standard quaternion for 90 deg rotation about z
            # RLBench uses [x, y, z, w] order
            # 90 deg about z: [0, 0, sin(π/4), cos(π/4)] = [0, 0, 0.7071, 0.7071]
            ninety_deg_quat = np.array([0, 0, 0.7071, 0.7071])
            print("[Task] Rotating gripper to 90 degrees (z axis).")
            obs, reward, done = rotate(
                env,
                task,
                target_quat=ninety_deg_quat,
                max_steps=100,
                threshold=0.05,
                timeout=10.0
            )
            if done:
                print("[Task] Task ended after rotate!")
                return
        except Exception as e:
            print(f"[Error] Failed to rotate gripper: {e}")
            shutdown_environment(env)
            return

        # --- Step 3: Pull the bottom drawer open ---
        try:
            # Pull along x axis (assume drawer opens along +x), pull distance ~0.15m
            pull_distance = 0.15
            print("[Task] Pulling the bottom drawer open along x axis.")
            obs, reward, done = pull(
                env,
                task,
                pull_distance=pull_distance,
                pull_axis='x',
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            if done:
                print("[Task] Task ended after pull!")
                return
        except Exception as e:
            print(f"[Error] Failed to pull the drawer: {e}")
            shutdown_environment(env)
            return

        # --- Step 4: Move to the trash (rubbish/item3) on the table ---
        try:
            # Identify which item is trash. Assume 'item3' is trash (per goal).
            item3_pos = positions.get('item3', None)
            if item3_pos is None:
                raise KeyError("item3 (trash) not found in object positions.")
            print("[Task] Moving to trash (item3) position:", item3_pos)
            obs, reward, done = move(
                env,
                task,
                target_pos=np.array(item3_pos),
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            if done:
                print("[Task] Task ended after move to trash!")
                return
        except Exception as e:
            print(f"[Error] Failed to move to trash: {e}")
            shutdown_environment(env)
            return

        # --- Step 5: Pick up the trash (item3) ---
        try:
            print("[Task] Picking up the trash (item3).")
            obs, reward, done = pick(
                env,
                task,
                target_pos=np.array(item3_pos),
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Task] Task ended after pick!")
                return
        except Exception as e:
            print(f"[Error] Failed to pick up the trash: {e}")
            shutdown_environment(env)
            return

        # --- Step 6: Move to the bin ---
        try:
            bin_pos = positions.get('bin', None)
            if bin_pos is None:
                raise KeyError("bin not found in object positions.")
            print("[Task] Moving to bin position:", bin_pos)
            obs, reward, done = move(
                env,
                task,
                target_pos=np.array(bin_pos),
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            if done:
                print("[Task] Task ended after move to bin!")
                return
        except Exception as e:
            print(f"[Error] Failed to move to bin: {e}")
            shutdown_environment(env)
            return

        # --- Step 7: Place the trash in the bin ---
        try:
            print("[Task] Placing the trash (item3) in the bin.")
            obs, reward, done = place(
                env,
                task,
                target_pos=np.array(bin_pos),
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Task] Task completed successfully! Reward:", reward)
            else:
                print("[Task] Task not completed yet (done=False).")
        except Exception as e:
            print(f"[Error] Failed to place the trash in the bin: {e}")
            shutdown_environment(env)
            return

        print("[Task] Finished all plan steps. Other objects left untouched as required.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()