import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Task: Slide open any available drawer to access its contents, then chuck away any rubbish remaining on the table.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Step 1: Move gripper to the side position of an available drawer ===
        # We'll choose the 'bottom' drawer (unlocked and present in the environment)
        # Use 'bottom_side_pos' as the approach position for the gripper
        try:
            side_pos = positions['bottom_side_pos']
        except KeyError:
            print("[Error] 'bottom_side_pos' not found in object positions.")
            shutdown_environment(env)
            return

        print("[Task] Moving to bottom drawer side position:", side_pos)
        obs, reward, done = move(
            env,
            task,
            target_pos=side_pos,
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after move to drawer side position!")
            return

        # === Step 2: Rotate gripper to 90 degrees (for drawer handle alignment) ===
        # We'll use a quaternion for 90 degrees rotation about z axis
        # Assuming initial orientation is zero_deg, so rotate to ninety_deg
        from scipy.spatial.transform import Rotation as R
        quat_90deg = R.from_euler('z', 90, degrees=True).as_quat()  # xyzw
        print("[Task] Rotating gripper to 90 degrees (drawer alignment)")
        obs, reward, done = rotate(
            env,
            task,
            target_quat=quat_90deg,
            max_steps=100,
            threshold=0.05,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after rotate!")
            return

        # === Step 3: Move gripper to anchor position (ready to pull) ===
        try:
            anchor_pos = positions['bottom_anchor_pos']
        except KeyError:
            print("[Error] 'bottom_anchor_pos' not found in object positions.")
            shutdown_environment(env)
            return

        print("[Task] Moving to bottom drawer anchor position:", anchor_pos)
        obs, reward, done = move(
            env,
            task,
            target_pos=anchor_pos,
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after move to anchor position!")
            return

        # === Step 4: Pull the drawer open ===
        # We'll pull along the x-axis by a reasonable distance (e.g., 0.15m)
        pull_distance = 0.15
        print("[Task] Pulling the drawer open along x-axis by", pull_distance)
        obs, reward, done = pull(
            env,
            task,
            pull_distance=pull_distance,
            pull_axis='x',
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after pull!")
            return

        # === Step 5: Move gripper above the table to pick up rubbish ===
        # We'll use 'item3' as the rubbish (as per the object list and goal)
        try:
            item3_pos = positions['item3']
        except KeyError:
            print("[Error] 'item3' not found in object positions.")
            shutdown_environment(env)
            return

        # Approach above the item (z+0.10 for safe approach)
        approach_item3_pos = np.array(item3_pos) + np.array([0, 0, 0.10])
        print("[Task] Moving above item3 (rubbish) at:", approach_item3_pos)
        obs, reward, done = move(
            env,
            task,
            target_pos=approach_item3_pos,
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after move above item3!")
            return

        # === Step 6: Pick up the rubbish (item3) ===
        print("[Task] Picking up item3 (rubbish) at:", item3_pos)
        obs, reward, done = pick(
            env,
            task,
            target_pos=item3_pos,
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after pick!")
            return

        # === Step 7: Place the rubbish in the bin ===
        try:
            bin_pos = positions['bin']
        except KeyError:
            print("[Error] 'bin' not found in object positions.")
            shutdown_environment(env)
            return

        # Approach above the bin (z+0.10 for safe approach)
        approach_bin_pos = np.array(bin_pos) + np.array([0, 0, 0.10])
        print("[Task] Moving above bin at:", approach_bin_pos)
        obs, reward, done = move(
            env,
            task,
            target_pos=approach_bin_pos,
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after move above bin!")
            return

        print("[Task] Placing item3 (rubbish) in bin at:", bin_pos)
        obs, reward, done = place(
            env,
            task,
            target_pos=bin_pos,
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task completed successfully! Reward:", reward)
        else:
            print("[Task] Task not completed yet (done=False).")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()