import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only the provided skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # --- Object name mapping (from object list and environment) ---
        # Drawer: choose 'bottom' drawer for this plan
        # Anchor/side/joint positions for bottom drawer
        bottom_anchor_pos = positions.get('bottom_anchor_pos', None)
        bottom_side_pos = positions.get('bottom_side_pos', None)
        bottom_joint_pos = positions.get('bottom_joint_pos', None)
        # Rubbish and bin
        rubbish_pos = positions.get('rubbish', None)
        bin_pos = positions.get('bin', None)
        # For initial approach, use a safe waypoint if available
        waypoint1_pos = positions.get('waypoint1', None)

        # === Plan Execution ===
        # Step 1: Rotate gripper to 90 degrees (for drawer manipulation)
        try:
            print("[Step 1] Rotating gripper to 90 degrees for drawer manipulation.")
            obs = task.get_observation()
            # Get current position and orientation
            start_pos = obs.gripper_pose[:3]
            start_quat = obs.gripper_pose[3:7]
            from scipy.spatial.transform import Rotation as R
            target_rot = R.from_euler('z', 90, degrees=True)
            target_quat = target_rot.as_quat()  # xyzw
            obs, reward, done = rotate(env, task, target_quat)
            if done:
                print("[Task] Task ended after rotate!")
                return
        except Exception as e:
            print(f"[ERROR] Step 1 (rotate) failed: {e}")
            shutdown_environment(env)
            return

        # Step 2: Move gripper to side position of bottom drawer
        try:
            print("[Step 2] Moving gripper to side position of bottom drawer.")
            if bottom_side_pos is None:
                raise ValueError("bottom_side_pos not found in object positions.")
            obs, reward, done = move(env, task, np.array(bottom_side_pos))
            if done:
                print("[Task] Task ended after move to side position!")
                return
        except Exception as e:
            print(f"[ERROR] Step 2 (move to side) failed: {e}")
            shutdown_environment(env)
            return

        # Step 3: Move gripper to anchor position of bottom drawer
        try:
            print("[Step 3] Moving gripper to anchor position of bottom drawer.")
            if bottom_anchor_pos is None:
                raise ValueError("bottom_anchor_pos not found in object positions.")
            obs, reward, done = move(env, task, np.array(bottom_anchor_pos))
            if done:
                print("[Task] Task ended after move to anchor position!")
                return
        except Exception as e:
            print(f"[ERROR] Step 3 (move to anchor) failed: {e}")
            shutdown_environment(env)
            return

        # Step 4: Pick (grasp) the drawer handle at anchor position
        try:
            print("[Step 4] Picking (grasping) the drawer handle at anchor position.")
            obs, reward, done = pick(env, task, np.array(bottom_anchor_pos), approach_distance=0.10, approach_axis='z')
            if done:
                print("[Task] Task ended after pick (drawer handle)!")
                return
        except Exception as e:
            print(f"[ERROR] Step 4 (pick drawer handle) failed: {e}")
            shutdown_environment(env)
            return

        # Step 5: Pull the drawer open (along x axis, positive direction)
        try:
            print("[Step 5] Pulling the drawer open.")
            pull_distance = 0.18
            obs, reward, done = pull(env, task, pull_distance, pull_axis='x')
            if done:
                print("[Task] Task ended after pull!")
                return
        except Exception as e:
            print(f"[ERROR] Step 5 (pull) failed: {e}")
            shutdown_environment(env)
            return

        # Step 6: Pick up the rubbish from the table (now accessible)
        try:
            print("[Step 6] Picking up the rubbish object.")
            if rubbish_pos is None:
                raise ValueError("rubbish position not found in object positions.")
            obs, reward, done = pick(env, task, np.array(rubbish_pos), approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended after pick (rubbish)!")
                return
        except Exception as e:
            print(f"[ERROR] Step 6 (pick rubbish) failed: {e}")
            shutdown_environment(env)
            return

        # Step 7: Place the rubbish in the bin
        try:
            print("[Step 7] Placing the rubbish in the bin.")
            if bin_pos is None:
                raise ValueError("bin position not found in object positions.")
            obs, reward, done = place(env, task, np.array(bin_pos), approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task completed successfully! Reward:", reward)
            else:
                print("[Task] Task not completed yet (done=False).")
        except Exception as e:
            print(f"[ERROR] Step 7 (place rubbish) failed: {e}")
            shutdown_environment(env)
            return

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()