import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use provided skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Object names from object list:
        # [
        # 'bottom_anchor_pos', 'bottom_joint_pos', 'bottom_side_pos',
        # 'middle_anchor_pos', 'middle_joint_pos', 'middle_side_pos',
        # 'top_anchor_pos', 'top_joint_pos', 'top_side_pos', 'waypoint1',
        # 'rubbish', 'tomato1', 'tomato2', 'bin'
        # ]

        # --- Plan: Open one of the drawers, then put the rubbish in the bin ---

        # We'll use the 'bottom' drawer for simplicity.
        # Step 1: Rotate gripper to 90 degrees (vertical for side approach)
        # Step 2: Move to the side position of the bottom drawer
        # Step 3: Move to the anchor position of the bottom drawer
        # Step 4: Pick (grasp) the drawer handle at anchor position
        # Step 5: Pull the drawer open
        # Step 6: Pick the rubbish from the table
        # Step 7: Place the rubbish in the bin

        # --- Step 1: Rotate gripper to 90 degrees (vertical) ---
        try:
            # Get quaternion for 90 degrees rotation about z axis
            # Assuming 'zero_deg' is [0,0,0,1] and 'ninety_deg' is a 90deg rotation about z
            # We'll use scipy to get the quaternion for 90deg about z
            from scipy.spatial.transform import Rotation as R
            ninety_deg_quat = R.from_euler('z', 90, degrees=True).as_quat()  # xyzw
            obs, reward, done = rotate(env, task, target_quat=ninety_deg_quat)
            if done:
                print("[Task] Task ended after rotate!")
                return
        except Exception as e:
            print(f"[ERROR] Step 1 (rotate) failed: {e}")
            return

        # --- Step 2: Move to side position of bottom drawer ---
        try:
            bottom_side_pos = positions.get('bottom_side_pos')
            if bottom_side_pos is None:
                raise ValueError("bottom_side_pos not found in positions.")
            obs, reward, done = move(env, task, target_pos=bottom_side_pos)
            if done:
                print("[Task] Task ended after move to side pos!")
                return
        except Exception as e:
            print(f"[ERROR] Step 2 (move to side) failed: {e}")
            return

        # --- Step 3: Move to anchor position of bottom drawer ---
        try:
            bottom_anchor_pos = positions.get('bottom_anchor_pos')
            if bottom_anchor_pos is None:
                raise ValueError("bottom_anchor_pos not found in positions.")
            obs, reward, done = move(env, task, target_pos=bottom_anchor_pos)
            if done:
                print("[Task] Task ended after move to anchor pos!")
                return
        except Exception as e:
            print(f"[ERROR] Step 3 (move to anchor) failed: {e}")
            return

        # --- Step 4: Pick (grasp) the drawer handle at anchor position ---
        try:
            # For picking the drawer, approach along -y or -z depending on drawer orientation.
            # We'll use -y as a common approach for horizontal drawers.
            obs, reward, done = pick(
                env,
                task,
                target_pos=bottom_anchor_pos,
                approach_distance=0.10,
                max_steps=100,
                threshold=0.01,
                approach_axis='-y',
                timeout=10.0
            )
            if done:
                print("[Task] Task ended after pick drawer handle!")
                return
        except Exception as e:
            print(f"[ERROR] Step 4 (pick drawer handle) failed: {e}")
            return

        # --- Step 5: Pull the drawer open ---
        try:
            # Pull along the x axis (assuming drawer opens along x)
            # Typical pull distance for a drawer: 0.15~0.20 meters
            obs, reward, done = pull(
                env,
                task,
                pull_distance=0.18,
                pull_axis='x',
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            if done:
                print("[Task] Task ended after pull!")
                return
        except Exception as e:
            print(f"[ERROR] Step 5 (pull) failed: {e}")
            return

        # --- Step 6: Pick the rubbish from the table ---
        try:
            rubbish_pos = positions.get('rubbish')
            if rubbish_pos is None:
                raise ValueError("rubbish not found in positions.")
            # Approach from above (z axis)
            obs, reward, done = pick(
                env,
                task,
                target_pos=rubbish_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Task] Task ended after pick rubbish!")
                return
        except Exception as e:
            print(f"[ERROR] Step 6 (pick rubbish) failed: {e}")
            return

        # --- Step 7: Place the rubbish in the bin ---
        try:
            bin_pos = positions.get('bin')
            if bin_pos is None:
                raise ValueError("bin not found in positions.")
            # Approach from above (z axis)
            obs, reward, done = place(
                env,
                task,
                target_pos=bin_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Task] Task completed successfully! Reward:", reward)
            else:
                print("[Task] Task not completed yet (done=False).")
        except Exception as e:
            print(f"[ERROR] Step 7 (place rubbish) failed: {e}")
            return

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
