import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use provided skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Object names from object list:
        # [
        # 'bottom_anchor_pos', 'bottom_joint_pos', 'bottom_side_pos',
        # 'middle_anchor_pos', 'middle_joint_pos', 'middle_side_pos',
        # 'top_anchor_pos', 'top_joint_pos', 'top_side_pos', 'waypoint1',
        # 'rubbish', 'tomato1', 'tomato2', 'bin'
        # ]

        # === Plan Execution ===
        # Goal: Select a drawer and open it fully, then pick up the rubbish and leave it in the trash can.

        # For simplicity, select the 'bottom' drawer.
        # Plan:
        # 1. Rotate gripper to 90 degrees (for side approach)
        # 2. Move to side position of bottom drawer
        # 3. Move to anchor position of bottom drawer
        # 4. Pick the drawer handle (anchor)
        # 5. Pull the drawer open
        # 6. Pick the rubbish from the table
        # 7. Place the rubbish in the bin

        # Step 1: Rotate gripper to 90 degrees (z axis)
        try:
            print("[Task] Step 1: Rotate gripper to 90 degrees for drawer approach.")
            # Get quaternion for 90 degrees rotation about z axis
            from scipy.spatial.transform import Rotation as R
            quat_ninety_deg = R.from_euler('z', 90, degrees=True).as_quat()  # xyzw
            obs, reward, done = rotate(env, task, target_quat=quat_ninety_deg)
            if done:
                print("[Task] Task ended after rotate!")
                return
        except Exception as e:
            print(f"[Task] Exception during rotate: {e}")
            shutdown_environment(env)
            return

        # Step 2: Move to side position of bottom drawer
        try:
            print("[Task] Step 2: Move to side position of bottom drawer.")
            bottom_side_pos = positions.get('bottom_side_pos')
            if bottom_side_pos is None:
                raise ValueError("bottom_side_pos not found in positions.")
            obs, reward, done = move(env, task, target_pos=bottom_side_pos)
            if done:
                print("[Task] Task ended after move to side position!")
                return
        except Exception as e:
            print(f"[Task] Exception during move to side: {e}")
            shutdown_environment(env)
            return

        # Step 3: Move to anchor position of bottom drawer
        try:
            print("[Task] Step 3: Move to anchor position of bottom drawer.")
            bottom_anchor_pos = positions.get('bottom_anchor_pos')
            if bottom_anchor_pos is None:
                raise ValueError("bottom_anchor_pos not found in positions.")
            obs, reward, done = move(env, task, target_pos=bottom_anchor_pos)
            if done:
                print("[Task] Task ended after move to anchor position!")
                return
        except Exception as e:
            print(f"[Task] Exception during move to anchor: {e}")
            shutdown_environment(env)
            return

        # Step 4: Pick the drawer handle (anchor)
        try:
            print("[Task] Step 4: Pick the drawer handle (anchor).")
            # Approach along -y or -z as appropriate; here use 'z' for top-down
            obs, reward, done = pick(env, task, target_pos=bottom_anchor_pos, approach_distance=0.10, approach_axis='z')
            if done:
                print("[Task] Task ended after pick drawer handle!")
                return
        except Exception as e:
            print(f"[Task] Exception during pick drawer handle: {e}")
            shutdown_environment(env)
            return

        # Step 5: Pull the drawer open
        try:
            print("[Task] Step 5: Pull the drawer open.")
            # Pull along x axis (assume drawer opens along +x)
            obs, reward, done = pull(env, task, pull_distance=0.18, pull_axis='x')
            if done:
                print("[Task] Task ended after pull!")
                return
        except Exception as e:
            print(f"[Task] Exception during pull: {e}")
            shutdown_environment(env)
            return

        # Step 6: Pick the rubbish from the table (assume rubbish is on table)
        try:
            print("[Task] Step 6: Pick the rubbish from the table.")
            rubbish_pos = positions.get('rubbish')
            if rubbish_pos is None:
                raise ValueError("rubbish not found in positions.")
            obs, reward, done = pick(env, task, target_pos=rubbish_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended after pick rubbish!")
                return
        except Exception as e:
            print(f"[Task] Exception during pick rubbish: {e}")
            shutdown_environment(env)
            return

        # Step 7: Place the rubbish in the bin
        try:
            print("[Task] Step 7: Place the rubbish in the bin.")
            bin_pos = positions.get('bin')
            if bin_pos is None:
                raise ValueError("bin not found in positions.")
            obs, reward, done = place(env, task, target_pos=bin_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task completed successfully! Reward:", reward)
            else:
                print("[Task] Task not completed yet (done=False).")
        except Exception as e:
            print(f"[Task] Exception during place rubbish: {e}")
            shutdown_environment(env)
            return

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()