import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use provided skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Task: Open a drawer to unlock the cabinet, then throw away the rubbish, leaving other objects alone.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Object names from object list and likely mapping:
        # 'bottom_anchor_pos', 'bottom_joint_pos', 'bottom_side_pos', 'middle_anchor_pos', 'middle_joint_pos', 'middle_side_pos', 'top_anchor_pos', 'top_joint_pos', 'top_side_pos', 'waypoint1', 'rubbish', 'tomato1', 'tomato2', 'bin'
        # We'll use: bottom_side_pos, bottom_anchor_pos, rubbish, bin

        # Defensive: check all required keys
        required_keys = [
            'bottom_side_pos', 'bottom_anchor_pos', 'rubbish', 'bin'
        ]
        for k in required_keys:
            if k not in positions:
                raise KeyError(f"Required object position '{k}' not found in get_object_positions().")

        # Step 1: rotate(gripper, zero_deg, ninety_deg)
        # We need to rotate the gripper to 90 degrees.
        # Let's assume zero_deg = [0,0,0,1] (identity quaternion), ninety_deg = [0,0,sin(45°),cos(45°)] = [0,0,0.7071,0.7071]
        # But in RLBench, the actual quaternion for 90deg about z is [0,0,0.7071,0.7071] (xyzw).
        # We'll get the current orientation and rotate to 90deg about z.
        # For robustness, let's define the target quaternion for 90deg about z.
        from scipy.spatial.transform import Rotation as R
        ninety_deg_quat = R.from_euler('z', 90, degrees=True).as_quat()  # xyzw

        print("[Plan Step 1] Rotating gripper to 90 degrees about z axis.")
        # [Frozen Code Start]
        obs, reward, done = rotate(
                env,
                task,
                target_quat=ninety_deg_quat,
                max_steps=100,
                threshold=0.05,
                timeout=10.0
            )
        # [Frozen Code End]
        if done:
            print("[Task] Task ended during rotate!")
            return

        # Step 2: move(gripper, side-pos-bottom)
        print("[Plan Step 2] Moving gripper to bottom_side_pos.")
        try:
            target_pos = positions['bottom_side_pos']
            obs, reward, done = move(
                env,
                task,
                target_pos=target_pos,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            if done:
                print("[Task] Task ended during move to side-pos-bottom!")
                return
        except Exception as e:
            print(f"[Error] Exception during move to side-pos-bottom: {e}")
            shutdown_environment(env)
            raise

        # Step 3: move(gripper, anchor-pos-bottom)
        print("[Plan Step 3] Moving gripper to bottom_anchor_pos.")
        try:
            target_pos = positions['bottom_anchor_pos']
            obs, reward, done = move(
                env,
                task,
                target_pos=target_pos,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            if done:
                print("[Task] Task ended during move to anchor-pos-bottom!")
                return
        except Exception as e:
            print(f"[Error] Exception during move to anchor-pos-bottom: {e}")
            shutdown_environment(env)
            raise

        # Step 4: pick(gripper, bottom)
        # In the domain, this is "pick-drawer", but only "pick" is available as a skill.
        # We'll interpret this as picking the drawer handle at anchor position.
        # We'll use the anchor position as the target for the pick.
        print("[Plan Step 4] Picking the bottom drawer handle at anchor position.")
        try:
            obs, reward, done = pick(
                env,
                task,
                target_pos=positions['bottom_anchor_pos'],
                approach_distance=0.10,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Task] Task ended during pick of bottom drawer!")
                return
        except Exception as e:
            print(f"[Error] Exception during pick of bottom drawer: {e}")
            shutdown_environment(env)
            raise

        # Step 5: pull(gripper, bottom)
        # We need to pull the drawer open. We'll pull along the x axis by a reasonable distance (e.g., 0.15m).
        print("[Plan Step 5] Pulling the bottom drawer open.")
        try:
            pull_distance = 0.15  # meters, adjust as needed for your environment
            obs, reward, done = pull(
                env,
                task,
                pull_distance=pull_distance,
                pull_axis='x',
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            if done:
                print("[Task] Task ended during pull of bottom drawer!")
                return
        except Exception as e:
            print(f"[Error] Exception during pull of bottom drawer: {e}")
            shutdown_environment(env)
            raise

        # Step 6: pick(rubbish, table)
        # We need to pick up the rubbish from the table.
        print("[Plan Step 6] Picking up the rubbish from the table.")
        try:
            obs, reward, done = pick(
                env,
                task,
                target_pos=positions['rubbish'],
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Task] Task ended during pick of rubbish!")
                return
        except Exception as e:
            print(f"[Error] Exception during pick of rubbish: {e}")
            shutdown_environment(env)
            raise

        # Step 7: place(rubbish, bin)
        # Place the rubbish in the bin.
        print("[Plan Step 7] Placing the rubbish in the bin.")
        try:
            obs, reward, done = place(
                env,
                task,
                target_pos=positions['bin'],
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Task] Task completed successfully! Reward:", reward)
            else:
                print("[Task] Task not completed yet (done=False).")
        except Exception as e:
            print(f"[Error] Exception during place of rubbish: {e}")
            shutdown_environment(env)
            raise

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()