# run_skeleton_task.py (Completed with Exploration Phase and Predicate Discovery)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def is_object_within_reach(gripper_pos, object_pos, reach_threshold=0.25):
    """Check if the object is within reach of the gripper."""
    dist = np.linalg.norm(np.array(gripper_pos) - np.array(object_pos))
    return dist <= reach_threshold

def log(msg):
    print(f"[LOG] {msg}")

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Example: positions = {'drawer': (x, y, z), 'tomato': (x, y, z), ...}

        # === Update Object List ===
        # Ensure all declared objects are present in the positions dictionary
        declared_objects = list(positions.keys())
        log(f"Declared objects in environment: {declared_objects}")

        # === Exploration Phase: Discover Missing Predicate ===
        # The feedback and exploration domain suggest that the robot may need to discover
        # a missing predicate (e.g., lock-known, identified, etc.) before performing certain actions.
        # We'll attempt to "move" to each object location and check for new information.

        # For demonstration, let's try to "move" to each object and log what we discover.
        # We'll also check reachability and log the process.

        # Assume the robot starts at some initial location; get from obs if possible
        try:
            gripper_pos = obs.gripper_pose[:3]
        except Exception:
            gripper_pos = np.zeros(3)
        log(f"Initial gripper position: {gripper_pos}")

        # For each object, perform exploration
        for obj_name, obj_pos in positions.items():
            log(f"Exploring object: {obj_name} at position {obj_pos}")

            # 1. Check reachability
            if not is_object_within_reach(gripper_pos, obj_pos):
                log(f"Object {obj_name} is not within reach. Attempting to move closer.")
                try:
                    # Use the move skill if available
                    obs, reward, done = move(env, task, target_pos=obj_pos, max_steps=50, threshold=0.01, timeout=5.0)
                    gripper_pos = obj_pos  # Update gripper position after move
                    log(f"Moved to {obj_name}.")
                except Exception as e:
                    log(f"Failed to move to {obj_name}: {e}")
                    continue
            else:
                log(f"Object {obj_name} is within reach.")

            # 2. Try to pick the object (if possible)
            try:
                obs, reward, done = pick(env, task, target_pos=obj_pos, approach_distance=0.15, max_steps=100, threshold=0.01, approach_axis='z', timeout=10.0)
                log(f"Picked up {obj_name}.")
                # After pick, check for new predicates (e.g., weight-known, durability-known)
                # In a real system, you would check the state here.
            except Exception as e:
                log(f"Could not pick {obj_name}: {e}")

            # 3. Place the object back (if possible)
            try:
                obs, reward, done = place(env, task, target_pos=obj_pos, approach_distance=0.15, max_steps=100, threshold=0.01, approach_axis='z', timeout=10.0)
                log(f"Placed {obj_name} back.")
            except Exception as e:
                log(f"Could not place {obj_name}: {e}")

            # 4. Try to rotate or pull if the object is a drawer or similar
            if 'drawer' in obj_name:
                # Try to rotate
                try:
                    # For demonstration, use a fixed target quaternion (identity)
                    target_quat = np.array([0, 0, 0, 1])
                    obs, reward, done = rotate(env, task, target_quat, max_steps=50, threshold=0.05, timeout=5.0)
                    log(f"Rotated {obj_name}.")
                except Exception as e:
                    log(f"Could not rotate {obj_name}: {e}")
                # Try to pull
                try:
                    obs, reward, done = pull(env, task, target_pos=obj_pos, max_steps=50, threshold=0.01, timeout=5.0)
                    log(f"Pulled {obj_name}.")
                except Exception as e:
                    log(f"Could not pull {obj_name}: {e}")

            # 5. Log completion for this object
            log(f"Finished exploration for {obj_name}.")

        # === Feedback and Logging for Task Completion ===
        log("Exploration phase complete. If a missing predicate was required, it should now be discovered.")
        log("Check logs above for any errors or missing information.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
