# run_skeleton_task.py (Completed with Exploration and Robustness)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

import traceback
import time
from typing import Dict, Any

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration and robust error handling.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'object_1': (0,0,0), 'object_2': (1,1,1), ...}
        try:
            positions: Dict[str, Any] = get_object_positions()
            print("[Info] Object positions loaded:", positions)
        except Exception as e:
            print("[Error] Failed to get object positions:", e)
            positions = {}

        # === Validate Object List ===
        try:
            object_list = task.get_objects() if hasattr(task, 'get_objects') else list(positions.keys())
            print("[Info] Objects in task:", object_list)
        except Exception as e:
            print("[Warning] Could not retrieve object list from task:", e)
            object_list = list(positions.keys())

        # === Exploration Phase: Identify Missing Predicate ===
        # The exploration domain suggests that some knowledge predicates (e.g., weight-known, durability-known, lock-known) may be missing.
        # We'll attempt to "explore" by moving to each location and picking each object, logging what is learned.

        # For demonstration, we assume 'locations' and 'objects' are available from positions or object_list.
        # This phase is generic and can be adapted to your specific environment.

        # --- Step 1: Move to each location and identify objects ---
        explored_objects = set()
        for obj_name, obj_pos in positions.items():
            print(f"[Exploration] Attempting to move to object '{obj_name}' at position {obj_pos}...")
            try:
                # Use 'move' skill if available; here, we assume move(env, task, target_pos, ...) signature
                obs, reward, done = move(
                    env,
                    task,
                    target_pos=obj_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                print(f"[Exploration] Moved to '{obj_name}'.")
                explored_objects.add(obj_name)
                if done:
                    print("[Exploration] Task ended during exploration move!")
                    return
            except Exception as e:
                print(f"[Error] Failed to move to '{obj_name}': {e}")
                traceback.print_exc()
                continue

        # --- Step 2: Pick each object to learn weight/durability ---
        for obj_name in explored_objects:
            print(f"[Exploration] Attempting to pick object '{obj_name}'...")
            try:
                obj_pos = positions[obj_name]
                obs, reward, done = pick(
                    env,
                    task,
                    target_pos=obj_pos,
                    approach_distance=0.10,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                print(f"[Exploration] Picked '{obj_name}'.")
                # After pick, try to place it back (if place skill exists)
                if 'place' in globals():
                    obs, reward, done = place(
                        env,
                        task,
                        target_pos=obj_pos,
                        approach_distance=0.10,
                        max_steps=100,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=10.0
                    )
                    print(f"[Exploration] Placed '{obj_name}' back.")
                if done:
                    print("[Exploration] Task ended during exploration pick/place!")
                    return
            except Exception as e:
                print(f"[Error] Failed to pick/place '{obj_name}': {e}")
                traceback.print_exc()
                continue

        # --- Step 3: Try to pull drawers/handles if present ---
        for obj_name in explored_objects:
            if 'drawer' in obj_name or 'handle' in obj_name:
                print(f"[Exploration] Attempting to pull '{obj_name}'...")
                try:
                    obj_pos = positions[obj_name]
                    obs, reward, done = pull(
                        env,
                        task,
                        target_pos=obj_pos,
                        approach_distance=0.10,
                        max_steps=100,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=10.0
                    )
                    print(f"[Exploration] Pulled '{obj_name}'.")
                    if done:
                        print("[Exploration] Task ended during exploration pull!")
                        return
                except Exception as e:
                    print(f"[Error] Failed to pull '{obj_name}': {e}")
                    traceback.print_exc()
                    continue

        # --- Step 4: Try to rotate gripper if required (for drawers etc.) ---
        if 'rotate' in globals():
            try:
                # Example: rotate to 90 degrees (quaternion for 90 deg about z)
                from scipy.spatial.transform import Rotation as R
                target_quat = R.from_euler('z', 90, degrees=True).as_quat()  # xyzw
                obs, reward, done = rotate(
                    env,
                    task,
                    target_quat=target_quat,
                    max_steps=100,
                    threshold=0.05,
                    timeout=10.0
                )
                print("[Exploration] Rotated gripper to 90 degrees.")
                if done:
                    print("[Exploration] Task ended during exploration rotate!")
                    return
            except Exception as e:
                print(f"[Error] Failed to rotate gripper: {e}")
                traceback.print_exc()

        # === Main Task Plan (Oracle Plan Execution) ===
        # After exploration, proceed with the main plan (example logic below).
        # Replace with your actual oracle plan steps as needed.

        print("[Task] Starting main oracle plan execution...")

        # Example: For each object, pick and place to a target location (if such a plan is required)
        for obj_name in explored_objects:
            print(f"[Task] Executing plan for '{obj_name}'...")
            try:
                obj_pos = positions[obj_name]
                # Pick
                obs, reward, done = pick(
                    env,
                    task,
                    target_pos=obj_pos,
                    approach_distance=0.10,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                print(f"[Task] Picked '{obj_name}'.")
                if done:
                    print("[Task] Task ended after pick!")
                    return
                # Place (to a predefined location, e.g., disposal bin)
                if 'place' in globals() and 'disposal_bin' in positions:
                    bin_pos = positions['disposal_bin']
                    obs, reward, done = place(
                        env,
                        task,
                        target_pos=bin_pos,
                        approach_distance=0.10,
                        max_steps=100,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=10.0
                    )
                    print(f"[Task] Placed '{obj_name}' in disposal bin.")
                    if done:
                        print("[Task] Task ended after place!")
                        return
            except Exception as e:
                print(f"[Error] Failed during main plan for '{obj_name}': {e}")
                traceback.print_exc()
                continue

        print("[Task] Oracle plan execution complete.")

    except Exception as main_e:
        print("[Fatal Error] Exception during skeleton task execution:", main_e)
        traceback.print_exc()
    finally:
        # Always ensure the environment is properly shutdown
        try:
            shutdown_environment(env)
        except Exception as shutdown_e:
            print("[Warning] Exception during environment shutdown:", shutdown_e)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
