# run_skeleton_task.py (Completed Executable)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'drawer1': ..., 'object_1': ..., ...}
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate (is-open drawer1) ===
        # Feedback indicates (is-open drawer1) is missing; we need to explore to determine this.
        # We use only available skills: ['pick', 'place', 'move', 'rotate', 'pull']
        # The only skill that can affect or check the open state is likely 'pull' (for drawers).
        # So, we attempt to pull the drawer and observe the result.

        # --- Identify drawer and gripper/position info ---
        # These names may need to be adapted to your environment's naming conventions.
        drawer_name = 'drawer1'
        gripper_name = 'gripper1' if 'gripper1' in positions else None
        drawer_pos = positions.get(drawer_name, None)
        gripper_pos = positions.get(gripper_name, None)

        # For demonstration, we assume the drawer handle is at drawer1's position.
        # In a real setup, you may need to use a specific handle position.

        # Step 1: Move gripper to drawer handle position (approach for pick-drawer)
        if drawer_pos is not None:
            print(f"[Exploration] Moving gripper to drawer handle at: {drawer_pos}")
            try:
                obs, reward, done = pick(
                    env,
                    task,
                    target_pos=drawer_pos,
                    approach_distance=0.10,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print("[Exploration] Task ended during pick (drawer handle)!")
                    return
            except Exception as e:
                print(f"[Exploration] Exception during pick: {e}")
        else:
            print("[Exploration] Drawer position not found in object positions.")
            return

        # Step 2: Attempt to pull the drawer open (this will help us discover is-open)
        print(f"[Exploration] Attempting to pull the drawer '{drawer_name}' to check open state.")
        try:
            obs, reward, done = pull(
                env,
                task,
                drawer_name
            )
            if done:
                print("[Exploration] Task ended during pull!")
                return
        except Exception as e:
            print(f"[Exploration] Exception during pull: {e}")

        # At this point, the exploration phase should have revealed the missing predicate (is-open drawer1)
        # and the environment should now reflect the drawer's open state.

        # === Main Task Plan (Example) ===
        # If the goal is to pick an object from inside the drawer, continue with pick/place as needed.
        # For demonstration, we show a generic pick and place if objects are available.

        # Example: pick an object (e.g., 'object_1') and place it at a target location (e.g., 'target_zone')
        object_to_pick = None
        for obj_name in positions:
            if obj_name.startswith('object') or obj_name.startswith('item'):
                object_to_pick = obj_name
                break

        if object_to_pick is not None:
            obj_pos = positions[object_to_pick]
            print(f"[Task] Picking up object: {object_to_pick} at {obj_pos}")
            try:
                obs, reward, done = pick(
                    env,
                    task,
                    target_pos=obj_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print("[Task] Task ended after pick!")
                    return
            except Exception as e:
                print(f"[Task] Exception during pick: {e}")

            # Place at a target location if available
            target_location = None
            for loc_name in positions:
                if 'target' in loc_name or 'zone' in loc_name:
                    target_location = positions[loc_name]
                    break
            if target_location is not None:
                print(f"[Task] Placing object at: {target_location}")
                try:
                    obs, reward, done = place(
                        env,
                        task,
                        target_pos=target_location,
                        approach_distance=0.15,
                        max_steps=100,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=10.0
                    )
                    if done:
                        print("[Task] Task ended after place!")
                        return
                except Exception as e:
                    print(f"[Task] Exception during place: {e}")
            else:
                print("[Task] No target location found for placing the object.")
        else:
            print("[Task] No pickable object found in object positions.")

        # === End of Task ===

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()