# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'item3': ..., 'drawer0': ..., ...}
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (is-side-pos approach_above_item3 drawer0) is missing.
        # We need to explore and check if the gripper is at a side position for drawer0.
        # Since we cannot directly check predicates, we attempt to move the gripper to the approach position
        # and see if the subsequent skill (e.g., move-to-side or pick-drawer) fails or succeeds.

        # For this example, let's assume:
        # - 'item3' is the object to manipulate
        # - 'drawer0' is the drawer to interact with
        # - 'approach_above_item3' is a named position above item3 (from positions dict)
        # - The plan is to move above item3, then interact with the drawer

        # Extract required positions from the positions dictionary
        try:
            approach_pos = positions['approach_above_item3']
        except KeyError:
            print("[Exploration] approach_above_item3 position not found in positions dictionary.")
            approach_pos = None

        try:
            drawer_pos = positions['drawer0']
        except KeyError:
            print("[Exploration] drawer0 position not found in positions dictionary.")
            drawer_pos = None

        # Step 1: Move gripper to approach position above item3
        if approach_pos is not None:
            print("[Exploration] Moving gripper to approach_above_item3 position:", approach_pos)
            try:
                obs, reward, done = move(
                    env,
                    task,
                    target_pos=np.array(approach_pos),
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                if done:
                    print("[Exploration] Task ended during move to approach_above_item3!")
                    return
            except Exception as e:
                print("[Exploration] Exception during move to approach_above_item3:", e)
                return
        else:
            print("[Exploration] Skipping move to approach_above_item3 due to missing position.")

        # Step 2: Try to pick item3 (if available)
        if 'item3' in positions:
            item3_pos = positions['item3']
            print("[Exploration] Attempting to pick item3 at position:", item3_pos)
            try:
                obs, reward, done = pick(
                    env,
                    task,
                    target_pos=np.array(item3_pos),
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print("[Exploration] Task ended after picking item3!")
                    return
            except Exception as e:
                print("[Exploration] Exception during pick of item3:", e)
        else:
            print("[Exploration] item3 not found in positions dictionary.")

        # Step 3: Try to rotate the gripper (if required by the plan)
        # For demonstration, let's assume we need to rotate to 'ninety_deg' (if such an angle is defined)
        # Since we don't have angle values, this is a placeholder for the rotate skill
        # Uncomment and adjust if angle values are available in positions or elsewhere
        # try:
        #     obs, reward, done = rotate(env, task, gripper_name, from_angle, to_angle)
        #     if done:
        #         print("[Exploration] Task ended after rotate!")
        #         return
        # except Exception as e:
        #     print("[Exploration] Exception during rotate:", e)

        # Step 4: Try to pull the drawer (if required)
        if drawer_pos is not None:
            print("[Exploration] Attempting to pull drawer0 at position:", drawer_pos)
            try:
                obs, reward, done = pull(
                    env,
                    task,
                    target_pos=np.array(drawer_pos),
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                if done:
                    print("[Exploration] Task ended after pulling drawer0!")
                    return
            except Exception as e:
                print("[Exploration] Exception during pull of drawer0:", e)
        else:
            print("[Exploration] Skipping pull of drawer0 due to missing position.")

        # Step 5: Place item3 at a target location (if required)
        # For demonstration, let's assume a 'target_location' is defined
        if 'target_location' in positions:
            target_location = positions['target_location']
            print("[Exploration] Attempting to place item3 at target_location:", target_location)
            try:
                obs, reward, done = place(
                    env,
                    task,
                    target_pos=np.array(target_location),
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print("[Exploration] Task ended after placing item3!")
                    return
            except Exception as e:
                print("[Exploration] Exception during place of item3:", e)
        else:
            print("[Exploration] target_location not found in positions dictionary.")

        # === End of Exploration Phase ===
        print("[Exploration] Exploration phase completed. If any action failed, check for missing predicates such as (is-side-pos approach_above_item3 drawer0).")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()