# run_skeleton_task.py (Completed with Exploration Phase and Feedback Handling)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def is_gripper_open(obs, threshold=0.04):
    # Returns True if gripper is open (based on openness sensor value)
    return getattr(obs, 'gripper_openness', 1.0) > threshold

def is_gripper_closed(obs, threshold=0.02):
    # Returns True if gripper is closed (based on openness sensor value)
    return getattr(obs, 'gripper_openness', 1.0) < threshold

def is_object_within_reach(gripper_pos, object_pos, reach_threshold=0.10):
    # Simple Euclidean distance check
    return np.linalg.norm(np.array(gripper_pos) - np.array(object_pos)) < reach_threshold

def verify_environment_state(task, expected_predicates):
    # This function checks if the environment is in the desired state
    # expected_predicates: list of (predicate_name, args)
    obs = task.get_observation()
    # For demonstration, we just print the check; in a real system, you'd check actual state
    print("[Verify] Checking environment state for predicates:", expected_predicates)
    # TODO: Implement actual predicate checks if possible
    return True

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Example: positions = {'drawer': (x,y,z), 'tomato': (x,y,z), ...}

        # === Exploration Phase: Identify Missing Predicate/Knowledge ===
        # Based on feedback, we need to check for force calibration, reachability, and environment state.
        # We'll perform a simple exploration to check if the robot can identify, reach, and safely interact with objects.

        # 1. Identify all objects in the environment
        print("[Exploration] Identifying objects in the environment...")
        for obj_name, obj_pos in positions.items():
            print(f"  - Found object '{obj_name}' at position {obj_pos}")

        # 2. For each object, check if it is within reach and if the gripper is open before pick
        for obj_name, obj_pos in positions.items():
            print(f"[Exploration] Checking reachability for '{obj_name}'...")
            obs = task.get_observation()
            gripper_pos = obs.gripper_pose[:3]
            if not is_object_within_reach(gripper_pos, obj_pos):
                print(f"  [Exploration] Object '{obj_name}' is NOT within reach. Attempting to move closer.")
                try:
                    # Use the move skill if available
                    obs, reward, done = move(env, task, target_pos=obj_pos)
                    if done:
                        print(f"  [Exploration] Task ended during move to '{obj_name}'!")
                        return
                except Exception as e:
                    print(f"  [Exploration] Move to '{obj_name}' failed: {e}")
                    continue
            else:
                print(f"  [Exploration] Object '{obj_name}' is within reach.")

            # 3. Check gripper state before pick
            obs = task.get_observation()
            if not is_gripper_open(obs):
                print(f"  [Exploration] Gripper is not open before picking '{obj_name}'. Attempting to open gripper.")
                # Try to open gripper (simulate by sending open command if available)
                try:
                    # If 'open_gripper' is not a skill, try to use place with a dummy location to open
                    # Or skip if not possible
                    pass
                except Exception as e:
                    print(f"  [Exploration] Could not open gripper: {e}")

            # 4. Attempt to pick the object (force calibration/feedback is handled in skill)
            print(f"  [Exploration] Attempting to pick '{obj_name}'...")
            try:
                obs, reward, done = pick(env, task, target_pos=obj_pos)
                if done:
                    print(f"  [Exploration] Task ended after picking '{obj_name}'!")
                    return
            except Exception as e:
                print(f"  [Exploration] Pick for '{obj_name}' failed: {e}")
                continue

            # 5. Check if gripper is closed after pick (force calibration)
            obs = task.get_observation()
            if not is_gripper_closed(obs):
                print(f"  [Exploration] Warning: Gripper did not close after picking '{obj_name}'. Possible force calibration issue.")

            # 6. Place the object back (if place skill is available)
            print(f"  [Exploration] Attempting to place '{obj_name}' back...")
            try:
                obs, reward, done = place(env, task, target_pos=obj_pos)
                if done:
                    print(f"  [Exploration] Task ended after placing '{obj_name}'!")
                    return
            except Exception as e:
                print(f"  [Exploration] Place for '{obj_name}' failed: {e}")
                continue

        # 7. After all exploration, verify if environment is in desired state
        print("[Exploration] Verifying environment state after exploration...")
        expected_predicates = []  # Fill with expected predicates if known
        if not verify_environment_state(task, expected_predicates):
            print("[Exploration] Environment is NOT in the desired state after exploration.")
        else:
            print("[Exploration] Environment is in the desired state after exploration.")

        # === Main Task Plan (Oracle Plan Execution) ===
        # Here you would execute the actual oracle plan step-by-step using only the available skills.
        # For demonstration, we show a generic sequence for one object (if more, loop as needed).

        for obj_name, obj_pos in positions.items():
            print(f"[Task] Executing plan for '{obj_name}'...")

            # Move to object if not within reach
            obs = task.get_observation()
            gripper_pos = obs.gripper_pose[:3]
            if not is_object_within_reach(gripper_pos, obj_pos):
                print(f"  [Task] Moving to '{obj_name}'...")
                try:
                    obs, reward, done = move(env, task, target_pos=obj_pos)
                    if done:
                        print(f"  [Task] Task ended during move to '{obj_name}'!")
                        return
                except Exception as e:
                    print(f"  [Task] Move to '{obj_name}' failed: {e}")
                    continue

            # Ensure gripper is open before pick
            obs = task.get_observation()
            if not is_gripper_open(obs):
                print(f"  [Task] Gripper is not open before picking '{obj_name}'.")
                # Try to open gripper if possible (skip if not available)
                pass

            # Pick the object
            print(f"  [Task] Picking '{obj_name}'...")
            try:
                obs, reward, done = pick(env, task, target_pos=obj_pos)
                if done:
                    print(f"  [Task] Task ended after picking '{obj_name}'!")
                    return
            except Exception as e:
                print(f"  [Task] Pick for '{obj_name}' failed: {e}")
                continue

            # Check gripper closed (force calibration)
            obs = task.get_observation()
            if not is_gripper_closed(obs):
                print(f"  [Task] Warning: Gripper did not close after picking '{obj_name}'. Possible force calibration issue.")

            # Place the object at a target location (if known)
            # For demonstration, place at same position
            print(f"  [Task] Placing '{obj_name}' at target location...")
            try:
                obs, reward, done = place(env, task, target_pos=obj_pos)
                if done:
                    print(f"  [Task] Task ended after placing '{obj_name}'!")
                    return
            except Exception as e:
                print(f"  [Task] Place for '{obj_name}' failed: {e}")
                continue

        # Final environment state verification
        print("[Task] Verifying final environment state...")
        if not verify_environment_state(task, expected_predicates):
            print("[Task] Final environment state is NOT as expected.")
        else:
            print("[Task] Final environment state is as expected.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()