# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'drawer0': ..., 'gripper': ..., ...}
        positions = get_object_positions()

        # === Exploration Phase: Check for Missing Predicate (is-open drawer0) ===
        # Feedback indicates (is-open drawer0) is missing; we need to explore and check if the drawer is open.
        # We'll use the available skills to try to open the drawer if it's not open.

        # Assumptions for object names (adjust as needed based on your environment/object_positions):
        drawer_name = 'drawer0'
        gripper_name = 'gripper'
        # Positions for gripper and drawer
        try:
            drawer_pos = positions[drawer_name]
            gripper_pos = positions[gripper_name]
        except KeyError as e:
            print(f"[Error] Missing object in positions: {e}")
            shutdown_environment(env)
            return

        # Step 1: Move gripper to a position near the drawer (side position)
        print(f"[Exploration] Moving gripper to side of {drawer_name} at {drawer_pos}")
        try:
            obs, reward, done = move(env, task, target_pos=drawer_pos)
            if done:
                print("[Exploration] Task ended during move to drawer side!")
                return
        except Exception as e:
            print(f"[Error] Exception during move: {e}")
            shutdown_environment(env)
            return

        # Step 2: Rotate gripper to required angle (e.g., 90 degrees) for drawer manipulation
        # We assume the rotate skill takes (env, task, from_angle, to_angle)
        # For demonstration, let's try to rotate from 'zero_deg' to 'ninety_deg'
        print("[Exploration] Rotating gripper to ninety_deg")
        try:
            obs, reward, done = rotate(env, task, from_angle='zero_deg', to_angle='ninety_deg')
            if done:
                print("[Exploration] Task ended during rotate!")
                return
        except Exception as e:
            print(f"[Error] Exception during rotate: {e}")
            shutdown_environment(env)
            return

        # Step 3: Attempt to pull the drawer open
        print(f"[Exploration] Attempting to pull {drawer_name}")
        try:
            obs, reward, done = pull(env, task, drawer_name)
            if done:
                print("[Exploration] Task ended during pull!")
                return
        except Exception as e:
            print(f"[Error] Exception during pull: {e}")
            shutdown_environment(env)
            return

        # Step 4: Check if the drawer is open (simulate predicate check)
        # In a real system, you would check the environment state or observation for is-open
        # Here, we print feedback for demonstration
        print(f"[Exploration] Checked: (is-open {drawer_name}) should now be true (if pull succeeded)")

        # === End of Exploration Phase ===

        # === Continue with Oracle Plan or Task-Specific Actions ===
        # (Insert further plan steps here as needed, using only predefined skills.)

        print("[Task] Exploration and predicate check complete.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()