# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'object_1': (x,y,z), ...}
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback and exploration domain suggest that the robot may be missing knowledge about some object property (e.g., weight, durability, lock status, etc.)
        # We'll perform a sequence of actions to explore and try to discover which predicate is missing.
        # We use only the available skills: ['pick', 'place', 'move', 'rotate', 'pull']

        # 1. List all objects in the environment
        object_list = list(positions.keys())
        print("[Exploration] Objects detected in environment:", object_list)

        # 2. For each object, attempt to pick and place it, and try to pull if possible.
        #    We will log any errors or unexpected results to infer missing predicates.
        for obj_name in object_list:
            print(f"\n[Exploration] Attempting to interact with object: {obj_name}")
            target_pos = positions[obj_name]
            try:
                # Safety: Check if object is valid
                if obj_name not in object_list:
                    print(f"[Exploration] {obj_name} not in object list, skipping.")
                    continue

                # Safety: Check if target position is valid (not too close to origin or out of bounds)
                if np.any(np.abs(target_pos) > 10) or np.linalg.norm(target_pos) < 0.05:
                    print(f"[Exploration] {obj_name} position {target_pos} is out of safe bounds, skipping.")
                    continue

                # 3. Try to pick the object
                print(f"[Exploration] Trying to pick {obj_name} at {target_pos}")
                obs, reward, done = pick(
                    env,
                    task,
                    target_pos=target_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print(f"[Exploration] Task ended after picking {obj_name}!")
                    break

                # 4. Try to place the object back (if place skill is available)
                if 'place' in globals():
                    # For demonstration, place at a slightly offset position
                    place_pos = np.array(target_pos) + np.array([0.05, 0, 0])
                    print(f"[Exploration] Trying to place {obj_name} at {place_pos}")
                    obs, reward, done = place(
                        env,
                        task,
                        target_pos=place_pos,
                        approach_distance=0.15,
                        max_steps=100,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=10.0
                    )
                    if done:
                        print(f"[Exploration] Task ended after placing {obj_name}!")
                        break

                # 5. Try to pull the object (if pull skill is available)
                if 'pull' in globals():
                    print(f"[Exploration] Trying to pull {obj_name}")
                    try:
                        obs, reward, done = pull(
                            env,
                            task,
                            obj_name
                        )
                        if done:
                            print(f"[Exploration] Task ended after pulling {obj_name}!")
                            break
                    except Exception as e:
                        print(f"[Exploration] Pull failed for {obj_name}: {e}")

            except Exception as e:
                print(f"[Exploration] Exception while handling {obj_name}: {e}")
                continue

        # === Exploration Summary ===
        print("\n[Exploration] Exploration phase complete.")
        print("If any object could not be picked, placed, or pulled, check logs above for missing predicate clues.")
        print("Based on the exploration domain, possible missing predicates include: identified, temperature-known, weight-known, durability-known, lock-known.")

        # === End of Task ===

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
