# run_skeleton_task.py (Completed with Exploration Phase and Feedback Handling)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback and exploration domain suggest that some object properties (e.g., lock-known, identified, temperature-known, etc.)
        # may be missing and need to be discovered before certain actions can be performed.
        # We'll attempt to "explore" by moving to each location and interacting with objects to discover their properties.

        # For demonstration, let's assume we want to ensure that for each object, we know its lock state before pulling (for drawers).
        # We'll also check for gripper_openness and euler_from_quat as per feedback.

        # Helper: Check for gripper_openness in observation
        def get_gripper_openness(obs):
            if hasattr(obs, 'gripper_openness'):
                return obs.gripper_openness
            else:
                print("[Warning] gripper_openness not found in observation. Using default value 0.0.")
                return 0.0

        # Helper: euler_from_quat (feedback requires this)
        def euler_from_quat(quat):
            # Assumes quat is [x, y, z, w]
            import math
            x, y, z, w = quat
            t0 = +2.0 * (w * x + y * z)
            t1 = +1.0 - 2.0 * (x * x + y * y)
            roll_x = np.arctan2(t0, t1)
            t2 = +2.0 * (w * y - z * x)
            t2 = +1.0 if t2 > +1.0 else t2
            t2 = -1.0 if t2 < -1.0 else t2
            pitch_y = np.arcsin(t2)
            t3 = +2.0 * (w * z + x * y)
            t4 = +1.0 - 2.0 * (y * y + z * z)
            yaw_z = np.arctan2(t3, t4)
            return np.array([roll_x, pitch_y, yaw_z])

        # Helper: Normalize quaternion
        def normalize_quaternion(q):
            q = np.array(q)
            return q / np.linalg.norm(q)

        # === Main Exploration and Task Execution ===

        # Example: Suppose we have a list of drawers and want to pull them, but must ensure lock-known predicate is satisfied.
        # We'll move to each drawer, try to pick or interact, and if not possible, perform exploration actions.

        # For demonstration, let's assume positions contains keys like 'drawer1', 'drawer2', etc.
        # We'll also handle generic objects for pick/place.

        # List of objects to explore (drawers and generic objects)
        object_keys = list(positions.keys())
        print("[Exploration] Objects in environment:", object_keys)

        # For each object, try to perform exploration and then the main action
        for obj_key in object_keys:
            obj_pos = positions[obj_key]
            print(f"[Exploration] Processing object: {obj_key} at position {obj_pos}")

            # 1. Move to the object's location (if move skill is available)
            if 'move' in ['pick', 'place', 'move', 'rotate', 'pull']:
                try:
                    # For demonstration, assume move takes (env, task, target_pos, ...)
                    print(f"[Action] Moving to {obj_key} at {obj_pos}")
                    obs, reward, done = move(
                        env,
                        task,
                        target_pos=obj_pos,
                        approach_distance=0.10,
                        max_steps=100,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=10.0
                    )
                    if done:
                        print(f"[Task] Task ended after move to {obj_key}!")
                        return
                except Exception as e:
                    print(f"[Error] Move to {obj_key} failed: {e}")

            # 2. Try to pick the object (if pick skill is available)
            if 'pick' in ['pick', 'place', 'move', 'rotate', 'pull']:
                try:
                    print(f"[Action] Attempting to pick {obj_key}")
                    obs, reward, done = pick(
                        env,
                        task,
                        target_pos=obj_pos,
                        approach_distance=0.10,
                        max_steps=100,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=10.0
                    )
                    if done:
                        print(f"[Task] Task ended after pick of {obj_key}!")
                        return
                except Exception as e:
                    print(f"[Error] Pick of {obj_key} failed: {e}")

            # 3. If object is a drawer, try to rotate and pull (if those skills are available)
            if 'drawer' in obj_key.lower():
                # Try to rotate (if rotate skill is available)
                if 'rotate' in ['pick', 'place', 'move', 'rotate', 'pull']:
                    try:
                        print(f"[Action] Rotating gripper for {obj_key}")
                        # For demonstration, use a target quaternion (identity)
                        target_quat = [0, 0, 0, 1]
                        obs, reward, done = rotate(
                            env,
                            task,
                            target_quat=target_quat,
                            max_steps=100,
                            threshold=0.05,
                            timeout=10.0
                        )
                        if done:
                            print(f"[Task] Task ended after rotate for {obj_key}!")
                            return
                    except Exception as e:
                        print(f"[Error] Rotate for {obj_key} failed: {e}")

                # Try to pull (if pull skill is available)
                if 'pull' in ['pick', 'place', 'move', 'rotate', 'pull']:
                    try:
                        print(f"[Action] Attempting to pull {obj_key}")
                        obs, reward, done = pull(
                            env,
                            task,
                            target_pos=obj_pos,
                            approach_distance=0.10,
                            max_steps=100,
                            threshold=0.01,
                            approach_axis='z',
                            timeout=10.0
                        )
                        if done:
                            print(f"[Task] Task ended after pull of {obj_key}!")
                            return
                    except Exception as e:
                        print(f"[Error] Pull of {obj_key} failed: {e}")

            # 4. Try to place the object (if place skill is available)
            if 'place' in ['pick', 'place', 'move', 'rotate', 'pull']:
                try:
                    # For demonstration, place at a default location (e.g., disposal bin)
                    if 'disposal' in positions:
                        place_pos = positions['disposal']
                    else:
                        # If no disposal, just offset the current position
                        place_pos = np.array(obj_pos) + np.array([0.2, 0, 0])
                    print(f"[Action] Placing {obj_key} at {place_pos}")
                    obs, reward, done = place(
                        env,
                        task,
                        target_pos=place_pos,
                        approach_distance=0.10,
                        max_steps=100,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=10.0
                    )
                    if done:
                        print(f"[Task] Task ended after place of {obj_key}!")
                        return
                except Exception as e:
                    print(f"[Error] Place of {obj_key} failed: {e}")

        print("[Exploration] Completed exploration and attempted all actions for all objects.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()