# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        print("[Exploration] Retrieved object positions:", positions)

        # === Exploration Phase: Try all available skills to discover missing predicates ===
        # The goal is to execute each available skill in a safe, exploratory way to see which predicates are affected.
        # This helps identify which predicate is missing in the domain/problem.

        # List of available skills
        available_skills = ['pick', 'place', 'move', 'rotate', 'pull']

        # For exploration, we need to select objects and locations to use with the skills.
        # We'll try to pick the first available object and location for each action.
        # This is a generic exploration; in a real scenario, you may want to iterate over all objects/locations.

        # Find at least one object and one location for testing
        object_names = [name for name in positions.keys() if 'drawer' not in name and 'gripper' not in name and 'angle' not in name and 'position' not in name]
        location_names = [name for name in positions.keys() if 'location' in name or 'table' in name or 'bin' in name or 'tray' in name]
        drawer_names = [name for name in positions.keys() if 'drawer' in name]
        gripper_names = [name for name in positions.keys() if 'gripper' in name]
        angle_names = [name for name in positions.keys() if 'angle' in name]
        position_names = [name for name in positions.keys() if 'position' in name]

        # Fallback: If no explicit location, use any object as a location (for at predicates)
        if not location_names and object_names:
            location_names = object_names

        # Select first available for each type
        test_object = object_names[0] if object_names else None
        test_location = location_names[0] if location_names else None
        test_drawer = drawer_names[0] if drawer_names else None
        test_gripper = gripper_names[0] if gripper_names else None
        test_angle_from = angle_names[0] if angle_names else None
        test_angle_to = angle_names[1] if len(angle_names) > 1 else None
        test_position_from = position_names[0] if position_names else None
        test_position_to = position_names[1] if len(position_names) > 1 else None

        # Print what we found for debugging
        print(f"[Exploration] test_object: {test_object}")
        print(f"[Exploration] test_location: {test_location}")
        print(f"[Exploration] test_drawer: {test_drawer}")
        print(f"[Exploration] test_gripper: {test_gripper}")
        print(f"[Exploration] test_angle_from: {test_angle_from}")
        print(f"[Exploration] test_angle_to: {test_angle_to}")
        print(f"[Exploration] test_position_from: {test_position_from}")
        print(f"[Exploration] test_position_to: {test_position_to}")

        # 1. Try 'move' skill (move gripper to object or location)
        if 'move' in available_skills and test_object:
            try:
                print(f"[Exploration] Trying move to {test_object} position...")
                target_pos = positions[test_object]
                obs, reward, done = move(env, task, target_pos)
                if done:
                    print("[Exploration] Task ended during move!")
                    return
            except Exception as e:
                print(f"[Exploration] move failed: {e}")

        # 2. Try 'pick' skill (pick up object at location)
        if 'pick' in available_skills and test_object and test_location:
            try:
                print(f"[Exploration] Trying pick on {test_object} at {test_location}...")
                # For pick, we need to approach the object
                target_pos = positions[test_object]
                obs, reward, done = move(env, task, target_pos)
                if done:
                    print("[Exploration] Task ended during move before pick!")
                    return
                # Now try pick (assuming pick signature: env, task, target_pos, ...)
                obs, reward, done = pick(env, task, target_pos)
                if done:
                    print("[Exploration] Task ended during pick!")
                    return
            except Exception as e:
                print(f"[Exploration] pick failed: {e}")

        # 3. Try 'place' skill (place object at location)
        if 'place' in available_skills and test_location:
            try:
                print(f"[Exploration] Trying place at {test_location}...")
                target_pos = positions[test_location]
                obs, reward, done = move(env, task, target_pos)
                if done:
                    print("[Exploration] Task ended during move before place!")
                    return
                obs, reward, done = place(env, task, target_pos)
                if done:
                    print("[Exploration] Task ended during place!")
                    return
            except Exception as e:
                print(f"[Exploration] place failed: {e}")

        # 4. Try 'rotate' skill (rotate gripper from one angle to another)
        if 'rotate' in available_skills and test_gripper and test_angle_from and test_angle_to:
            try:
                print(f"[Exploration] Trying rotate {test_gripper} from {test_angle_from} to {test_angle_to}...")
                obs, reward, done = rotate(env, task, test_gripper, test_angle_from, test_angle_to)
                if done:
                    print("[Exploration] Task ended during rotate!")
                    return
            except Exception as e:
                print(f"[Exploration] rotate failed: {e}")

        # 5. Try 'pull' skill (pull a drawer or object)
        if 'pull' in available_skills and test_drawer and test_gripper:
            try:
                print(f"[Exploration] Trying pull {test_drawer} with {test_gripper}...")
                obs, reward, done = pull(env, task, test_gripper, test_drawer)
                if done:
                    print("[Exploration] Task ended during pull!")
                    return
            except Exception as e:
                print(f"[Exploration] pull failed: {e}")

        print("[Exploration] Exploration phase complete. Check logs and environment for predicate changes or errors indicating missing predicates.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()