# run_skeleton_task.py (Completed for Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'object_1': (0,0,0), 'object_2': (1,1,1), ...}
        positions = get_object_positions()

        # === Exploration Phase: Try to Identify Missing Predicate ===
        # Based on feedback, the planner times out, likely due to a missing predicate in the domain/problem.
        # The exploration domain suggests predicates like identified, temperature-known, weight-known, durability-known, lock-known.
        # We will attempt to use available skills to interact with objects and observe if any information is missing or not updated.

        # For demonstration, we will:
        # 1. Move to each object location (if possible)
        # 2. Attempt to pick each object (if possible)
        # 3. Attempt to pull (if possible)
        # 4. Observe and print any errors or missing information

        # Only use available skills: ['pick', 'place', 'move', 'rotate', 'pull']
        # We do not define new skills or predicates.

        # List all objects and their positions
        print("[Exploration] Objects and positions in the environment:")
        for obj_name, obj_pos in positions.items():
            print(f"    {obj_name}: {obj_pos}")

        # Try to move to each object's position and pick it up
        for obj_name, obj_pos in positions.items():
            print(f"\n[Exploration] Attempting to move to and pick: {obj_name}")
            try:
                # Move the gripper to the object's position
                obs, reward, done = move(
                    env,
                    task,
                    target_pos=np.array(obj_pos),
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                if done:
                    print(f"[Exploration] Task ended during move to {obj_name}!")
                    break

                # Attempt to pick the object
                obs, reward, done = pick(
                    env,
                    task,
                    target_pos=np.array(obj_pos),
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print(f"[Exploration] Task ended during pick of {obj_name}!")
                    break

                print(f"[Exploration] Successfully picked {obj_name}.")

                # Try to pull (if the object is a drawer or can be pulled)
                print(f"[Exploration] Attempting to pull {obj_name} (if applicable).")
                try:
                    obs, reward, done = pull(
                        env,
                        task,
                        target_pos=np.array(obj_pos),
                        max_steps=100,
                        threshold=0.01,
                        timeout=10.0
                    )
                    if done:
                        print(f"[Exploration] Task ended during pull of {obj_name}!")
                        break
                    print(f"[Exploration] Successfully pulled {obj_name}.")
                except Exception as e:
                    print(f"[Exploration] Pull not applicable or failed for {obj_name}: {e}")

                # Place the object back (if possible)
                print(f"[Exploration] Attempting to place {obj_name} back.")
                try:
                    obs, reward, done = place(
                        env,
                        task,
                        target_pos=np.array(obj_pos),
                        approach_distance=0.15,
                        max_steps=100,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=10.0
                    )
                    if done:
                        print(f"[Exploration] Task ended during place of {obj_name}!")
                        break
                    print(f"[Exploration] Successfully placed {obj_name}.")
                except Exception as e:
                    print(f"[Exploration] Place not applicable or failed for {obj_name}: {e}")

            except Exception as e:
                print(f"[Exploration] Exception during move/pick for {obj_name}: {e}")

        print("\n[Exploration] Finished exploration phase. If any actions failed or information was missing, check logs for clues about missing predicates (e.g., lock-known, identified, etc.).")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()