# run_skeleton_task.py (Completed with Exploration Phase and Plan Execution)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'drawer0': ..., 'pos_init': ..., ...}
        positions = get_object_positions()

        # --- Exploration Phase: Check for Missing Predicate ---
        # Feedback: (is-anchor-pos pos_init drawer0) is missing or needs to be checked.
        # We will attempt to check if pos_init is an anchor position for drawer0.
        # If not, we can try to infer or log this for debugging.

        drawer_name = 'drawer0'
        anchor_pos_name = 'pos_init'
        gripper_name = 'gripper0' if 'gripper0' in positions else None

        # Defensive: Check if required objects exist
        if drawer_name not in positions or anchor_pos_name not in positions:
            print(f"[Exploration] Required objects '{drawer_name}' or '{anchor_pos_name}' not found in positions.")
            return

        anchor_pos = positions[anchor_pos_name]
        drawer_pos = positions[drawer_name]
        if gripper_name:
            gripper_pos = positions[gripper_name]
        else:
            # Fallback: Use current gripper position from observation
            gripper_pos = obs.gripper_pose[:3]

        print(f"[Exploration] Checking if '{anchor_pos_name}' is anchor for '{drawer_name}'...")

        # In the absence of a direct skill to check is-anchor-pos, we attempt to move the gripper to pos_init and see if subsequent actions (pick-drawer, pull) succeed.
        # This is a form of active exploration.

        # === Move gripper to anchor position (pos_init) ===
        try:
            print(f"[Exploration] Moving gripper to '{anchor_pos_name}' at {anchor_pos}")
            obs, reward, done = move(env, task, target_pos=anchor_pos)
            if done:
                print("[Exploration] Task ended unexpectedly during move to anchor position.")
                return
        except Exception as e:
            print(f"[Exploration] Exception during move to anchor position: {e}")
            return

        # === Attempt to rotate gripper to required angle (e.g., 'ninety_deg') if needed ===
        # Since the rotate skill is available, but we don't have angle names/values, we skip unless required.

        # === Attempt to pick the drawer at anchor position ===
        # Since we do not have a pick-drawer skill, but the domain uses 'pick-drawer', and our available skills are only pick/place/move/rotate/pull,
        # we proceed to try 'pull' after moving to anchor position, as a proxy for exploration.

        # --- Exploration: Try to pull the drawer ---
        try:
            print(f"[Exploration] Attempting to pull '{drawer_name}' after moving to anchor position.")
            obs, reward, done = pull(env, task, drawer_name)
            if done:
                print("[Exploration] Task ended after pull attempt.")
                return
            print("[Exploration] Pull action executed. If drawer opened, pos_init is likely anchor position.")
        except Exception as e:
            print(f"[Exploration] Exception during pull: {e}")
            print("[Exploration] This may indicate that pos_init is not an anchor position or preconditions are not met.")

        # --- End of Exploration Phase ---

        # === Main Task Plan Execution ===
        # Here, you would execute the oracle plan using only the available skills.
        # For demonstration, let's assume the goal is to open the drawer by pulling it from the anchor position.

        # 1. Move to anchor position (already done above)
        # 2. (Optional) Rotate gripper if required (not enough info to do so)
        # 3. Pull the drawer

        # If the drawer is now open, the task is complete.
        # If not, log and exit.

        # (Optional) Check if drawer is open by querying the environment or observation
        # This depends on the observation structure; here we just print a message.

        print("[Task] Plan execution complete. Check simulation for drawer state.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()