# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Check for Missing Predicate ===
        # Feedback: (rotated gripper1 ninety_deg)
        # The missing predicate is likely (rotated gripper1 ninety_deg)
        # We need to ensure the gripper is rotated to ninety_deg before certain actions.

        # For demonstration, let's assume the following:
        # - gripper1 is the gripper to be rotated
        # - ninety_deg is a predefined quaternion or angle in the environment
        # - The object_positions module or the environment provides the necessary info

        # --- Example: Exploration to ensure (rotated gripper1 ninety_deg) holds ---

        # Retrieve gripper and angle info (names may need to be adapted to your env)
        gripper_name = 'gripper1'
        target_angle_name = 'ninety_deg'

        # Try to get the target quaternion for ninety_deg from positions or env
        try:
            # This assumes get_object_positions() provides quaternions for angles
            target_quat = positions.get(target_angle_name)
            if target_quat is None:
                # Fallback: define a quaternion for 90 degrees about z-axis
                # (w, x, y, z) = (cos(theta/2), 0, 0, sin(theta/2))
                theta = np.pi / 2
                target_quat = np.array([np.cos(theta/2), 0, 0, np.sin(theta/2)])
        except Exception as e:
            print(f"[Exploration] Could not retrieve target quaternion for {target_angle_name}: {e}")
            # Fallback: use default 90 deg about z
            theta = np.pi / 2
            target_quat = np.array([np.cos(theta/2), 0, 0, np.sin(theta/2)])

        # --- Ensure the gripper is rotated to ninety_deg ---
        print(f"[Exploration] Rotating {gripper_name} to {target_angle_name} (quaternion: {target_quat})")
        try:
            obs, reward, done = rotate(
                env,
                task,
                target_quat=target_quat,
                max_steps=100,
                threshold=0.05,
                timeout=10.0
            )
            if done:
                print("[Exploration] Task ended during rotation!")
                return
        except Exception as e:
            print(f"[Exploration] Exception during rotate: {e}")

        # --- Continue with the rest of the plan using available skills ---
        # Example: pick, move, place, pull, etc.
        # The actual plan would depend on the task, but here is a generic sequence:

        # Example: Pick up an object after rotating the gripper
        object_to_pick = 'object_1'
        pick_location = 'location_1'
        try:
            object_pos = positions.get(object_to_pick)
            if object_pos is None:
                print(f"[Task] Could not find position for {object_to_pick}, skipping pick.")
            else:
                print(f"[Task] Picking up {object_to_pick} at {pick_location} (pos: {object_pos})")
                obs, reward, done = pick(
                    env,
                    task,
                    target_pos=object_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print("[Task] Task ended after pick!")
                    return
        except Exception as e:
            print(f"[Task] Exception during pick: {e}")

        # Example: Place the object at a target location
        place_location = 'location_2'
        try:
            place_pos = positions.get(place_location)
            if place_pos is None:
                print(f"[Task] Could not find position for {place_location}, skipping place.")
            else:
                print(f"[Task] Placing {object_to_pick} at {place_location} (pos: {place_pos})")
                obs, reward, done = place(
                    env,
                    task,
                    target_pos=place_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print("[Task] Task ended after place!")
                    return
        except Exception as e:
            print(f"[Task] Exception during place: {e}")

        # Example: Pull a drawer (if part of the plan)
        drawer_name = 'drawer_1'
        try:
            print(f"[Task] Attempting to pull {drawer_name}")
            obs, reward, done = pull(
                env,
                task,
                drawer_name
            )
            if done:
                print("[Task] Task ended after pull!")
                return
        except Exception as e:
            print(f"[Task] Exception during pull: {e}")

        # --- End of plan ---

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()