# run_skeleton_task.py (Completed with Exploration Phase and Robustness)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

import traceback

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'object_1': (0,0,0), ...}
        positions = get_object_positions()

        # === Get object list from positions ===
        object_list = list(positions.keys())
        print("[Exploration] Detected objects in environment:", object_list)

        # === Exploration Phase: Check for Missing Predicate Knowledge ===
        # The feedback and exploration PDDL suggest that some predicates (e.g., identified, temperature-known, weight-known, durability-known, lock-known) may be missing in the initial state.
        # We will attempt to "explore" each object/location to see if any knowledge is missing, and log the process.

        # For demonstration, let's try to "identify" all objects by moving to their locations (if possible).
        # We'll also check for gripper and drawer objects for possible rotate/pull actions.

        # Helper: Find locations for each object (if available)
        # For this example, we assume positions dict contains all relevant objects and their positions.
        # We'll also check for special objects like 'gripper', 'drawer', 'bin', etc.

        # Dynamic threshold and timeout values (can be tuned)
        angle_threshold = 0.05
        rotate_timeout = 10.0

        # Track which predicates we have "explored"
        explored_predicates = set()

        # 1. Try to identify all objects by moving to their positions (if move skill is available)
        if 'move' in ['pick', 'place', 'move', 'rotate', 'pull']:
            for obj_name, obj_pos in positions.items():
                try:
                    print(f"[Exploration] Attempting to move to object '{obj_name}' at position {obj_pos}...")
                    # For this simulation, we assume move(env, task, target_pos, ...) signature
                    obs, reward, done = move(
                        env,
                        task,
                        target_pos=obj_pos,
                        approach_distance=0.15,
                        max_steps=100,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=5.0
                    )
                    print(f"[Exploration] Moved to '{obj_name}'.")
                    explored_predicates.add('identified')
                    if done:
                        print("[Exploration] Task ended during move!")
                        return
                except Exception as e:
                    print(f"[Exploration] Could not move to '{obj_name}': {e}")
                    traceback.print_exc()

        # 2. Try to pick up objects to learn weight/durability (if pick skill is available)
        if 'pick' in ['pick', 'place', 'move', 'rotate', 'pull']:
            for obj_name, obj_pos in positions.items():
                # Skip non-pickable objects (e.g., bin, drawer, gripper, positions)
                if any(x in obj_name for x in ['bin', 'drawer', 'gripper', 'pos', 'waypoint']):
                    continue
                try:
                    print(f"[Exploration] Attempting to pick up '{obj_name}' at position {obj_pos}...")
                    obs, reward, done = pick(
                        env,
                        task,
                        target_pos=obj_pos,
                        approach_distance=0.15,
                        max_steps=100,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=5.0
                    )
                    print(f"[Exploration] Picked up '{obj_name}'.")
                    explored_predicates.add('weight-known')
                    explored_predicates.add('durability-known')
                    if done:
                        print("[Exploration] Task ended during pick!")
                        return
                except Exception as e:
                    print(f"[Exploration] Could not pick up '{obj_name}': {e}")
                    traceback.print_exc()

        # 3. Try to rotate gripper if 'rotate' skill is available (for drawers, etc.)
        if 'rotate' in ['pick', 'place', 'move', 'rotate', 'pull']:
            # Try rotating to a known orientation (e.g., 90 degrees)
            try:
                # For demonstration, use a target quaternion (e.g., 90 deg about z)
                from scipy.spatial.transform import Rotation as R
                target_euler = [0, 0, np.pi/2]  # 90 deg about z
                target_quat = R.from_euler('xyz', target_euler).as_quat()  # xyzw
                print(f"[Exploration] Attempting to rotate gripper to {target_euler} (rad)...")
                obs, reward, done = rotate(
                    env,
                    task,
                    target_quat=target_quat,
                    max_steps=100,
                    threshold=angle_threshold,
                    timeout=rotate_timeout
                )
                print("[Exploration] Rotated gripper.")
                explored_predicates.add('rotated')
                if done:
                    print("[Exploration] Task ended during rotate!")
                    return
            except Exception as e:
                print(f"[Exploration] Could not rotate gripper: {e}")
                traceback.print_exc()

        # 4. Try to pull drawers if 'pull' skill is available
        if 'pull' in ['pick', 'place', 'move', 'rotate', 'pull']:
            for obj_name in object_list:
                if 'drawer' in obj_name:
                    try:
                        print(f"[Exploration] Attempting to pull '{obj_name}'...")
                        # For demonstration, assume pull(env, task, drawer_name, ...)
                        obs, reward, done = pull(
                            env,
                            task,
                            drawer_name=obj_name,
                            max_steps=100,
                            threshold=0.01,
                            timeout=5.0
                        )
                        print(f"[Exploration] Pulled '{obj_name}'.")
                        explored_predicates.add('is-open')
                        if done:
                            print("[Exploration] Task ended during pull!")
                            return
                    except Exception as e:
                        print(f"[Exploration] Could not pull '{obj_name}': {e}")
                        traceback.print_exc()

        # 5. Try to place objects in bin (if 'place' skill is available)
        if 'place' in ['pick', 'place', 'move', 'rotate', 'pull']:
            for obj_name, obj_pos in positions.items():
                if 'bin' in positions:
                    try:
                        print(f"[Exploration] Attempting to place '{obj_name}' in bin...")
                        obs, reward, done = place(
                            env,
                            task,
                            target_pos=positions['bin'],
                            approach_distance=0.15,
                            max_steps=100,
                            threshold=0.01,
                            approach_axis='z',
                            timeout=5.0
                        )
                        print(f"[Exploration] Placed '{obj_name}' in bin.")
                        if done:
                            print("[Exploration] Task ended during place!")
                            return
                    except Exception as e:
                        print(f"[Exploration] Could not place '{obj_name}' in bin: {e}")
                        traceback.print_exc()

        # === Exploration Summary ===
        print("[Exploration] Explored predicates:", explored_predicates)
        missing_predicates = {'identified', 'temperature-known', 'weight-known', 'durability-known', 'lock-known'} - explored_predicates
        if missing_predicates:
            print("[Exploration] Missing predicate knowledge after exploration:", missing_predicates)
        else:
            print("[Exploration] All key predicates explored.")

        # === End of Exploration Phase ===
        # At this point, you can proceed to the main oracle plan if needed.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()