# run_skeleton_task.py (Completed with Exploration Phase and Predicate Discovery)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'bin': ..., 'bottom': ..., ...}
        positions = get_object_positions()

        # === Exploration Phase: Discover Missing Predicate ===
        # Feedback: (is-side-pos bin bottom)
        # The predicate (is-side-pos bin bottom) is missing or needs to be confirmed.
        # We will attempt to explore the environment to check if 'bottom' is a side position of 'bin'.
        # This is a logical exploration, so we simulate the check and print the result.

        print("=== [Exploration] Checking missing predicate: (is-side-pos bin bottom) ===")
        bin_pos = positions.get('bin', None)
        bottom_pos = positions.get('bottom', None)
        if bin_pos is None or bottom_pos is None:
            print("[Exploration] Could not find 'bin' or 'bottom' in object positions. Exploration failed.")
        else:
            # Try to move to 'bottom' and see if it is a side position of 'bin'
            # In a real scenario, this would involve checking the environment or using sensors.
            # Here, we simulate the check and print the result.
            print(f"[Exploration] bin position: {bin_pos}")
            print(f"[Exploration] bottom position: {bottom_pos}")
            # For demonstration, we assume that if 'bottom' is close to 'bin', it is a side position.
            distance = np.linalg.norm(np.array(bin_pos) - np.array(bottom_pos))
            if distance < 0.2:  # Threshold for "side" (arbitrary, adjust as needed)
                print("[Exploration] (is-side-pos bin bottom) likely TRUE (distance < 0.2).")
            else:
                print("[Exploration] (is-side-pos bin bottom) likely FALSE (distance >= 0.2).")
            # This exploration step is for predicate discovery only.

        # === Main Task Plan ===
        # Example: Suppose the plan is to pick an object from 'bottom' and place it in 'bin'
        # Use only the available skills: ['pick', 'place', 'move', 'rotate', 'pull']

        # For demonstration, let's try to pick an object at 'bottom' and place it at 'bin'
        # You may need to adjust object names and positions according to your environment

        # Find an object at 'bottom' (other than 'bin' and 'bottom' themselves)
        object_to_pick = None
        for obj_name, obj_pos in positions.items():
            if obj_name not in ['bin', 'bottom']:
                # Assume objects at 'bottom' are to be picked
                # In a real scenario, check if obj_pos is close to bottom_pos
                if bottom_pos is not None and np.linalg.norm(np.array(obj_pos) - np.array(bottom_pos)) < 0.05:
                    object_to_pick = obj_name
                    break

        if object_to_pick is None:
            print("[Task] No object found at 'bottom' to pick. Skipping pick and place.")
        else:
            print(f"[Task] Will pick object '{object_to_pick}' at 'bottom' and place in 'bin'.")

            # Move to above the object to pick
            approach_height = 0.10  # 10cm above
            pick_pos = np.array(positions[object_to_pick])
            pick_approach_pos = pick_pos.copy()
            pick_approach_pos[2] += approach_height

            try:
                print(f"[Task] Moving above '{object_to_pick}' at {pick_approach_pos}")
                obs, reward, done = move(env, task, pick_approach_pos)
                if done:
                    print("[Task] Task ended during approach to pick position!")
                    return

                # Move down to pick position
                print(f"[Task] Moving down to pick '{object_to_pick}' at {pick_pos}")
                obs, reward, done = move(env, task, pick_pos)
                if done:
                    print("[Task] Task ended during move to pick position!")
                    return

                # Pick the object
                print(f"[Task] Picking '{object_to_pick}' at {pick_pos}")
                obs, reward, done = pick(env, task, target_pos=pick_pos)
                if done:
                    print("[Task] Task ended during pick!")
                    return

                # Move up after pick
                print(f"[Task] Moving up after pick to {pick_approach_pos}")
                obs, reward, done = move(env, task, pick_approach_pos)
                if done:
                    print("[Task] Task ended after pick!")
                    return

                # Move above bin
                bin_pos_arr = np.array(bin_pos)
                bin_approach_pos = bin_pos_arr.copy()
                bin_approach_pos[2] += approach_height

                print(f"[Task] Moving above 'bin' at {bin_approach_pos}")
                obs, reward, done = move(env, task, bin_approach_pos)
                if done:
                    print("[Task] Task ended during move to bin!")
                    return

                # Move down to bin
                print(f"[Task] Moving down to 'bin' at {bin_pos_arr}")
                obs, reward, done = move(env, task, bin_pos_arr)
                if done:
                    print("[Task] Task ended during move to bin position!")
                    return

                # Place the object
                print(f"[Task] Placing '{object_to_pick}' at 'bin'")
                obs, reward, done = place(env, task, target_pos=bin_pos_arr)
                if done:
                    print("[Task] Task ended after place!")
                    return

                # Move up after place
                print(f"[Task] Moving up after place to {bin_approach_pos}")
                obs, reward, done = move(env, task, bin_approach_pos)
                if done:
                    print("[Task] Task ended after place (move up)!")
                    return

                print("[Task] Pick and place completed successfully.")

            except Exception as e:
                print(f"[Task] Exception during pick and place: {e}")

        # === End of Main Task Plan ===

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()