# run_skeleton_task.py (Completed with Exploration Phase for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'gripper': ..., 'bin_pos': ..., ...}
        positions = get_object_positions()

        # --- Exploration Phase: Check for Missing Predicate (gripper-at gripper bin_pos) ---
        # The feedback indicates that the predicate (gripper-at gripper bin_pos) is missing.
        # We will attempt to move the gripper to 'bin_pos' and check if the environment reflects this.

        gripper_name = 'gripper'
        bin_pos_name = 'bin_pos'

        # Defensive: Check if required keys exist
        if gripper_name not in positions or bin_pos_name not in positions:
            print(f"[Exploration] Missing required object positions: '{gripper_name}' or '{bin_pos_name}' not found in positions dict.")
            print("[Exploration] Available keys:", list(positions.keys()))
            return

        gripper_pos = positions[gripper_name]
        bin_pos = positions[bin_pos_name]

        print(f"[Exploration] Moving gripper to bin_pos: {bin_pos}")

        try:
            # Use the provided 'move' skill to move the gripper to bin_pos
            obs, reward, done = move(
                env,
                task,
                target_pos=np.array(bin_pos),
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            print("[Exploration] Gripper moved to bin_pos.")
        except Exception as e:
            print(f"[Exploration] Exception during move: {e}")
            return

        # After moving, check if the predicate is now true (if possible)
        # Since we do not have direct access to the PDDL state, we can only log the action.
        # In a real system, you might query the environment or check the observation.

        # --- End of Exploration Phase ---

        # === (Optional) Continue with Task Plan ===
        # If the exploration is only for missing predicate, we can stop here.
        # Otherwise, you can continue with the rest of the oracle plan using available skills.

        # Example: (Commented out, as no further plan is specified)
        # print("[Task] Picking up object at bin_pos...")
        # obs, reward, done = pick(env, task, target_pos=np.array(bin_pos), ...)
        # if done:
        #     print("[Task] Task ended after pick!")
        #     return

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
