# run_skeleton_task.py (Completed with Exploration Phase for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'gripper': (x,y,z), 'bin': (x,y,z), ...}
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate (gripper-at gripper bin) ===
        # Feedback indicates (gripper-at gripper bin) is missing.
        # We need to explore and confirm the gripper's position at 'bin'.
        # We'll use the 'move' skill to move the gripper to the 'bin' position and check if the predicate holds.

        gripper_name = 'gripper'
        bin_name = 'bin'

        # Defensive: Check if required objects are present in positions
        if gripper_name not in positions:
            print(f"[Exploration] ERROR: '{gripper_name}' position not found in object positions.")
            return
        if bin_name not in positions:
            print(f"[Exploration] ERROR: '{bin_name}' position not found in object positions.")
            return

        gripper_pos = positions[gripper_name]
        bin_pos = positions[bin_name]

        print(f"[Exploration] Moving gripper to bin position: {bin_pos}")

        # Use the move skill to move the gripper to the bin
        try:
            obs, reward, done = move(
                env,
                task,
                target_pos=np.array(bin_pos),
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
        except Exception as e:
            print(f"[Exploration] Exception during move: {e}")
            return

        # After moving, check if the gripper is at the bin (predicate gripper-at gripper bin)
        # This is a logical check; in a real system, you might update a state tracker or log this.
        # For demonstration, we print the result.
        obs = task.get_observation()
        current_gripper_pos = obs.gripper_pose[:3]
        dist_to_bin = np.linalg.norm(np.array(bin_pos) - current_gripper_pos)
        if dist_to_bin < 0.02:
            print(f"[Exploration] SUCCESS: (gripper-at gripper bin) predicate is now TRUE (distance {dist_to_bin:.4f})")
        else:
            print(f"[Exploration] WARNING: (gripper-at gripper bin) predicate is FALSE (distance {dist_to_bin:.4f})")

        # === (Optional) Continue with Oracle Plan or Task Steps ===
        # If the plan requires further actions (e.g., pick, place, rotate, pull), add them here using the available skills.
        # For this exploration, we focus on the missing predicate.

        # Example (commented out, as not required for this exploration):
        # obs, reward, done = pick(env, task, target_pos=some_obj_pos, ...)
        # if done:
        #     print("[Task] Task ended after pick!")
        #     return

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
