# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'drawer0': ..., 'object_1': ..., ...}
        positions = get_object_positions()

        # --- Exploration Phase: Check for missing predicate (is-open drawer0) ---
        # The feedback indicates that (is-open drawer0) is a missing predicate.
        # We will attempt to explore the state of drawer0 using available skills.

        # Assumptions:
        # - 'drawer0' is the drawer to be explored.
        # - The gripper must move to the drawer handle, rotate if needed, and pull to open.
        # - The available skills are: move, pick, place, rotate, pull.

        # Step 1: Move gripper to the drawer handle position
        try:
            drawer_handle_pos = positions['drawer0_handle']
        except KeyError:
            print("[Exploration] drawer0_handle position not found in object_positions.")
            return

        print("[Exploration] Moving gripper to drawer0 handle position:", drawer_handle_pos)
        try:
            obs, reward, done = move(
                env,
                task,
                target_pos=np.array(drawer_handle_pos),
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            if done:
                print("[Exploration] Task ended during move to drawer handle!")
                return
        except Exception as e:
            print("[Exploration] Exception during move:", e)
            return

        # Step 2: Rotate gripper to required angle (assume 'ninety_deg' is required)
        # We need to know the current angle and the target angle.
        # For this example, we assume the angles are named 'zero_deg' and 'ninety_deg'.
        # If the environment provides angle names or values, use them accordingly.

        # For demonstration, we use string names as in the PDDL.
        gripper_name = 'gripper0'
        from_angle = 'zero_deg'
        to_angle = 'ninety_deg'
        print(f"[Exploration] Rotating gripper {gripper_name} from {from_angle} to {to_angle}")
        try:
            obs, reward, done = rotate(
                env,
                task,
                gripper=gripper_name,
                from_angle=from_angle,
                to_angle=to_angle
            )
            if done:
                print("[Exploration] Task ended during gripper rotation!")
                return
        except Exception as e:
            print("[Exploration] Exception during rotate:", e)
            return

        # Step 3: Pull the drawer to open it
        drawer_name = 'drawer0'
        print(f"[Exploration] Pulling drawer {drawer_name} with gripper {gripper_name}")
        try:
            obs, reward, done = pull(
                env,
                task,
                gripper=gripper_name,
                drawer=drawer_name
            )
            if done:
                print("[Exploration] Task ended during pull!")
                return
        except Exception as e:
            print("[Exploration] Exception during pull:", e)
            return

        # Step 4: Check if the drawer is open (predicate is-open drawer0)
        # Since we cannot directly check predicates, we assume the environment or observation
        # provides a way to verify if the drawer is open (e.g., via object_positions or obs).
        # For demonstration, we print a message.
        print("[Exploration] Exploration complete. (is-open drawer0) should now be true if pull succeeded.")

        # --- End of Exploration Phase ---

        # Additional task logic can be added here if needed.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()