# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'drawer1': ..., 'object_1': ..., ...}
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback indicates (is-open drawer1) is missing.
        # We need to explore the state of drawer1 to determine if it is open.
        # This can be done by attempting to pull the drawer and observing the result.

        # For this example, we assume the drawer is named 'drawer1' and the gripper is 'gripper1'.
        # The positions dictionary should contain the relevant positions.
        # If not, handle gracefully.

        drawer_name = 'drawer1'
        gripper_name = 'gripper1'
        anchor_pos_key = 'drawer1_anchor'
        side_pos_key = 'drawer1_side'
        angle_ninety = 'ninety_deg'
        angle_zero = 'zero_deg'

        # Defensive: Check if required keys exist
        if drawer_name not in positions or anchor_pos_key not in positions or side_pos_key not in positions:
            print("[Exploration] Required drawer or position keys not found in object_positions. Exploration aborted.")
            return

        drawer_pos = positions[drawer_name]
        anchor_pos = positions[anchor_pos_key]
        side_pos = positions[side_pos_key]

        # Step 1: Rotate gripper to 90 degrees (if not already)
        try:
            print("[Exploration] Rotating gripper to 90 degrees.")
            obs, reward, done = rotate(env, task, gripper_name, angle_zero, angle_ninety)
            if done:
                print("[Exploration] Task ended during rotate.")
                return
        except Exception as e:
            print(f"[Exploration] Exception during rotate: {e}")
            return

        # Step 2: Move gripper to side position of drawer
        try:
            print("[Exploration] Moving gripper to side position of drawer.")
            obs, reward, done = move(env, task, gripper_name, drawer_name, 'start', side_pos_key)
            if done:
                print("[Exploration] Task ended during move-to-side.")
                return
        except Exception as e:
            print(f"[Exploration] Exception during move-to-side: {e}")
            return

        # Step 3: Move gripper to anchor position of drawer
        try:
            print("[Exploration] Moving gripper to anchor position of drawer.")
            obs, reward, done = move(env, task, gripper_name, drawer_name, side_pos_key, anchor_pos_key)
            if done:
                print("[Exploration] Task ended during move-to-anchor.")
                return
        except Exception as e:
            print(f"[Exploration] Exception during move-to-anchor: {e}")
            return

        # Step 4: Pick the drawer handle
        try:
            print("[Exploration] Picking the drawer handle.")
            obs, reward, done = pick(env, task, target_pos=anchor_pos, approach_distance=0.10, max_steps=100, threshold=0.01, approach_axis='z', timeout=10.0)
            if done:
                print("[Exploration] Task ended during pick-drawer.")
                return
        except Exception as e:
            print(f"[Exploration] Exception during pick-drawer: {e}")
            return

        # Step 5: Attempt to pull the drawer to check if it is openable (is-open predicate)
        try:
            print("[Exploration] Attempting to pull the drawer.")
            obs, reward, done = pull(env, task, gripper_name, drawer_name)
            if done:
                print("[Exploration] Task ended during pull.")
                return
            # After pull, check if the drawer is open (simulate checking the predicate)
            # In a real system, you would check the state, but here we just print feedback.
            print("[Exploration] Pulled the drawer. If successful, (is-open drawer1) is now true.")
        except Exception as e:
            print(f"[Exploration] Exception during pull: {e}")
            return

        # === End of Exploration Phase ===

        # At this point, the missing predicate (is-open drawer1) should be discovered/achieved.

        # === Continue with Oracle Plan (if any) ===
        # Insert further plan steps here as needed, using only predefined skills.

        print("[Task] Exploration and predicate discovery complete.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
