# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'rubbish': (x, y, z), 'drawer': (x, y, z), ...}
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (rubbish_pos) is missing. We need to explore to find the position of 'rubbish'.
        # The exploration domain suggests that moving to a location allows us to identify objects there.
        # We'll attempt to move the gripper to all known positions and check for 'rubbish'.

        rubbish_pos = None
        found_rubbish = False

        # Try to find the rubbish by moving to each known object position
        for obj_name, pos in positions.items():
            print(f"[Exploration] Moving to {obj_name} at position {pos} to check for rubbish.")
            try:
                obs, reward, done = move(env, task, target_pos=np.array(pos))
            except Exception as e:
                print(f"[Exploration] Error moving to {obj_name}: {e}")
                continue

            # After moving, check if the current observation contains 'rubbish'
            # This assumes the observation has an attribute or method to get visible/nearby objects
            # If not, we use the feedback and exploration knowledge to assume the rubbish is at this position
            # For demonstration, we check if 'rubbish' is in the object name or if the position is plausible
            if 'rubbish' in obj_name.lower():
                rubbish_pos = pos
                found_rubbish = True
                print(f"[Exploration] Found rubbish at {rubbish_pos} (by name match).")
                break

        # If not found by name, try to guess based on feedback or default to a known position
        if not found_rubbish:
            # Try to find a plausible rubbish position by other means
            for obj_name, pos in positions.items():
                if 'trash' in obj_name.lower() or 'garbage' in obj_name.lower():
                    rubbish_pos = pos
                    found_rubbish = True
                    print(f"[Exploration] Found rubbish at {rubbish_pos} (by alternative name match).")
                    break

        if not found_rubbish:
            # As a last resort, pick the first position (not recommended, but prevents crash)
            rubbish_pos = list(positions.values())[0]
            print(f"[Exploration] Could not explicitly find rubbish. Defaulting to first object position: {rubbish_pos}")

        # === Main Task Plan ===
        # Now that we have rubbish_pos, we can proceed with the oracle plan using available skills.
        # Example: pick up the rubbish and place it in a disposal location (if known).

        # 1. Move to rubbish position (already done in exploration, but repeat for clarity)
        print(f"[Task] Moving to rubbish at {rubbish_pos}")
        try:
            obs, reward, done = move(env, task, target_pos=np.array(rubbish_pos))
            if done:
                print("[Task] Task ended during move to rubbish!")
                return
        except Exception as e:
            print(f"[Task] Error moving to rubbish: {e}")
            return

        # 2. Pick up the rubbish
        print("[Task] Attempting to pick up rubbish.")
        try:
            obs, reward, done = pick(
                env,
                task,
                target_pos=np.array(rubbish_pos),
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Task] Task ended after picking up rubbish!")
                return
        except Exception as e:
            print(f"[Task] Error picking up rubbish: {e}")
            return

        # 3. Place the rubbish in a disposal location if available
        # Try to find a disposal/trash location in positions
        disposal_pos = None
        for obj_name, pos in positions.items():
            if 'disposal' in obj_name.lower() or 'bin' in obj_name.lower() or 'trash' in obj_name.lower():
                disposal_pos = pos
                print(f"[Task] Found disposal location: {obj_name} at {disposal_pos}")
                break

        if disposal_pos is not None:
            print(f"[Task] Moving to disposal location at {disposal_pos}")
            try:
                obs, reward, done = move(env, task, target_pos=np.array(disposal_pos))
                if done:
                    print("[Task] Task ended during move to disposal!")
                    return
            except Exception as e:
                print(f"[Task] Error moving to disposal: {e}")
                return

            print("[Task] Placing rubbish in disposal location.")
            try:
                obs, reward, done = place(
                    env,
                    task,
                    target_pos=np.array(disposal_pos),
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print("[Task] Task ended after placing rubbish!")
                    return
            except Exception as e:
                print(f"[Task] Error placing rubbish: {e}")
                return
        else:
            print("[Task] No disposal location found. Skipping place action.")

        # === End of Task ===

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()