# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'gripper': ..., 'bin': ..., ...}
        positions = get_object_positions()

        # --- Exploration Phase: Identify Missing Predicate ---
        # Feedback: (gripper-at gripper bin)
        # The missing predicate is likely 'gripper-at', which indicates the gripper's position.
        # We need to explore and confirm the gripper's position at 'bin'.

        # Defensive: Check if required keys exist
        if 'gripper' not in positions or 'bin' not in positions:
            print("[Error] Required objects ('gripper', 'bin') not found in positions dictionary.")
            return

        gripper_pos = positions['gripper']
        bin_pos = positions['bin']

        print(f"[Exploration] Gripper initial position: {gripper_pos}")
        print(f"[Exploration] Bin position: {bin_pos}")

        # --- Move gripper to bin to establish (gripper-at gripper bin) ---
        try:
            print("[Action] Moving gripper to bin position to establish (gripper-at gripper bin)...")
            obs, reward, done = move(
                env,
                task,
                target_pos=bin_pos,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            if done:
                print("[Task] Task ended during move to bin!")
                return
        except Exception as e:
            print(f"[Error] Exception during move: {e}")
            return

        # --- Exploration: Confirm gripper is at bin (simulate predicate check) ---
        obs = task.get_observation()
        current_gripper_pos = obs.gripper_pose[:3]
        dist_to_bin = np.linalg.norm(current_gripper_pos - bin_pos)
        if dist_to_bin < 0.02:
            print("[Exploration] Confirmed: (gripper-at gripper bin) is now true.")
        else:
            print("[Exploration] Warning: Gripper not at bin as expected (distance: {:.4f})".format(dist_to_bin))

        # --- (Optional) Further exploration or skill execution can be added here ---
        # For example, pick or place actions if required by the plan.

        # Example: If the plan requires picking an object at the bin
        # if 'object_1' in positions:
        #     object_1_pos = positions['object_1']
        #     print("[Action] Picking object_1 at bin...")
        #     try:
        #         obs, reward, done = pick(
        #             env,
        #             task,
        #             target_pos=object_1_pos,
        #             approach_distance=0.15,
        #             max_steps=100,
        #             threshold=0.01,
        #             approach_axis='z',
        #             timeout=10.0
        #         )
        #         if done:
        #             print("[Task] Task ended after pick!")
        #             return
        #     except Exception as e:
        #         print(f"[Error] Exception during pick: {e}")
        #         return

        print("[Exploration] Exploration phase complete. Missing predicate identified and established.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()