# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'object_1': (0,0,0), 'object_2': (1,1,1), ...}
        positions = get_object_positions()

        # === Exploration Phase: Try to discover missing predicates ===
        # The feedback and exploration knowledge suggest that some predicates (e.g., lock-known, identified, temperature-known, etc.)
        # may be missing or not handled. We will attempt to use available skills to interact with objects and observe the environment,
        # looking for changes or information that may indicate missing predicates.

        # For demonstration, we will attempt to:
        # 1. Move to each object position (if possible)
        # 2. Pick and place objects (if possible)
        # 3. Rotate or pull drawers (if possible)
        # 4. Observe any errors, exceptions, or unexpected results that may indicate missing predicates

        # Only use available skills: ['pick', 'place', 'move', 'rotate', 'pull']
        # We do not define new skills.

        # List of objects and their positions
        object_names = list(positions.keys())
        print("[Exploration] Objects in environment:", object_names)

        # Track which actions succeed/fail for each object
        exploration_results = {}

        for obj_name in object_names:
            obj_pos = positions[obj_name]
            print(f"\n[Exploration] Attempting to interact with '{obj_name}' at position {obj_pos}")
            exploration_results[obj_name] = {}

            # 1. Try to move to the object's position
            try:
                print(f"[Exploration] Moving to '{obj_name}'...")
                obs, reward, done = move(env, task, target_pos=np.array(obj_pos))
                exploration_results[obj_name]['move'] = 'success'
                if done:
                    print(f"[Exploration] Task ended after move to '{obj_name}'!")
                    break
            except Exception as e:
                print(f"[Exploration] Move to '{obj_name}' failed: {e}")
                exploration_results[obj_name]['move'] = f'fail: {e}'
                continue  # If move fails, skip further actions for this object

            # 2. Try to pick the object
            try:
                print(f"[Exploration] Picking '{obj_name}'...")
                obs, reward, done = pick(
                    env,
                    task,
                    target_pos=np.array(obj_pos),
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                exploration_results[obj_name]['pick'] = 'success'
                if done:
                    print(f"[Exploration] Task ended after pick of '{obj_name}'!")
                    break
            except Exception as e:
                print(f"[Exploration] Pick of '{obj_name}' failed: {e}")
                exploration_results[obj_name]['pick'] = f'fail: {e}'

            # 3. Try to place the object back (if pick succeeded)
            try:
                print(f"[Exploration] Placing '{obj_name}'...")
                obs, reward, done = place(
                    env,
                    task,
                    target_pos=np.array(obj_pos),
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                exploration_results[obj_name]['place'] = 'success'
                if done:
                    print(f"[Exploration] Task ended after place of '{obj_name}'!")
                    break
            except Exception as e:
                print(f"[Exploration] Place of '{obj_name}' failed: {e}")
                exploration_results[obj_name]['place'] = f'fail: {e}'

            # 4. Try to rotate (if object is a drawer or has a handle)
            try:
                print(f"[Exploration] Rotating '{obj_name}' (if applicable)...")
                # For demonstration, try rotating from 0 to 90 degrees (if such angles exist)
                # You may need to adjust angle names/types based on your environment
                obs, reward, done = rotate(
                    env,
                    task,
                    gripper_name='gripper',  # Replace with actual gripper name if needed
                    from_angle='zero_deg',
                    to_angle='ninety_deg'
                )
                exploration_results[obj_name]['rotate'] = 'success'
                if done:
                    print(f"[Exploration] Task ended after rotate of '{obj_name}'!")
                    break
            except Exception as e:
                print(f"[Exploration] Rotate of '{obj_name}' failed: {e}")
                exploration_results[obj_name]['rotate'] = f'fail: {e}'

            # 5. Try to pull (if object is a drawer)
            try:
                print(f"[Exploration] Pulling '{obj_name}' (if applicable)...")
                obs, reward, done = pull(
                    env,
                    task,
                    drawer_name=obj_name  # Assuming the object is a drawer
                )
                exploration_results[obj_name]['pull'] = 'success'
                if done:
                    print(f"[Exploration] Task ended after pull of '{obj_name}'!")
                    break
            except Exception as e:
                print(f"[Exploration] Pull of '{obj_name}' failed: {e}")
                exploration_results[obj_name]['pull'] = f'fail: {e}'

        print("\n[Exploration] Summary of action results per object:")
        for obj, results in exploration_results.items():
            print(f"  {obj}: {results}")

        # === End of Exploration Phase ===

        # TODO: Based on the above exploration, analyze which actions fail and what predicates might be missing.
        # For example, if 'pull' always fails with a message about 'lock-known', then 'lock-known' is likely the missing predicate.
        # You can use this information to update your domain/problem files or inform further debugging.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()