# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'gripper': ..., 'bin': ..., ...}
        positions = get_object_positions()

        # --- Exploration Phase: Identify Missing Predicate ---
        # Feedback: (gripper-at gripper bin)
        # The predicate (gripper-at gripper bin) is missing or needs to be explored.
        # We will use the available 'move' skill to move the gripper to the 'bin' position,
        # and check if the predicate is satisfied (i.e., gripper is at bin).

        # Get the gripper and bin positions from the environment
        try:
            gripper_pos = positions['gripper']
            bin_pos = positions['bin']
        except KeyError as e:
            print(f"[Error] Missing object in positions: {e}")
            shutdown_environment(env)
            return

        print(f"[Exploration] Moving gripper to bin position: {bin_pos}")

        # Use the predefined 'move' skill to move the gripper to the bin
        try:
            obs, reward, done = move(
                env,
                task,
                target_pos=np.array(bin_pos),
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
        except Exception as e:
            print(f"[Error] Exception during move: {e}")
            shutdown_environment(env)
            return

        # After moving, check if the gripper is at the bin (predicate exploration)
        obs = task.get_observation()
        current_gripper_pos = obs.gripper_pose[:3]
        dist_to_bin = np.linalg.norm(np.array(bin_pos) - current_gripper_pos)
        if dist_to_bin < 0.02:
            print("[Exploration] Predicate (gripper-at gripper bin) is now TRUE (within threshold).")
        else:
            print("[Exploration] Predicate (gripper-at gripper bin) is FALSE (not at bin).")

        # --- End of Exploration Phase ---

        # === (Optional) Continue with Oracle Plan or Task Steps ===
        # If there is a further plan, continue with pick/place/rotate/pull as needed.
        # For this exploration, we focus on the missing predicate only.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
