# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'drawer1': ..., 'object_1': ..., ...}
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (is-open drawer1) is missing; we need to explore to determine if drawer1 is open.
        # We will use the available skills to explore the state of drawer1.

        # Assumptions:
        # - 'drawer1' is the name of the drawer object in the environment.
        # - positions contains the relevant positions for the drawer and possibly its handle.
        # - The robot must move to the drawer, rotate gripper if needed, and attempt to pull to check if it is open.

        # Step 1: Move to the drawer's handle position (side position)
        try:
            drawer_name = 'drawer1'
            gripper_name = 'gripper'  # If needed for skill functions
            # Try to get the side position for the drawer
            if 'drawer1_handle_side' in positions:
                drawer_side_pos = positions['drawer1_handle_side']
            elif 'drawer1' in positions:
                drawer_side_pos = positions['drawer1']
            else:
                raise KeyError("drawer1 or drawer1_handle_side not found in positions dictionary.")
        except Exception as e:
            print(f"[Error] Could not find drawer1 position: {e}")
            shutdown_environment(env)
            return

        print(f"[Exploration] Moving to drawer1 side position at {drawer_side_pos}")
        try:
            obs, reward, done = move(
                env,
                task,
                target_pos=np.array(drawer_side_pos),
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            if done:
                print("[Exploration] Task ended during move to drawer1!")
                return
        except Exception as e:
            print(f"[Error] Exception during move to drawer1: {e}")
            shutdown_environment(env)
            return

        # Step 2: Rotate gripper to required angle (e.g., 90 degrees) if needed
        # Assume angle names: 'zero_deg', 'ninety_deg'
        # If the rotate skill requires angle names, use them; otherwise, skip if not needed.
        try:
            print("[Exploration] Rotating gripper to ninety_deg for side approach.")
            obs, reward, done = rotate(
                env,
                task,
                gripper_name,
                'zero_deg',
                'ninety_deg'
            )
            if done:
                print("[Exploration] Task ended during gripper rotation!")
                return
        except Exception as e:
            print(f"[Warning] Could not rotate gripper: {e} (may not be required)")

        # Step 3: (Optional) Move to anchor position if required by the plan
        # If anchor position is needed for pick-drawer, move there
        if 'drawer1_handle_anchor' in positions:
            drawer_anchor_pos = positions['drawer1_handle_anchor']
            print(f"[Exploration] Moving to drawer1 anchor position at {drawer_anchor_pos}")
            try:
                obs, reward, done = move(
                    env,
                    task,
                    target_pos=np.array(drawer_anchor_pos),
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                if done:
                    print("[Exploration] Task ended during move to anchor!")
                    return
            except Exception as e:
                print(f"[Error] Exception during move to anchor: {e}")
                shutdown_environment(env)
                return

        # Step 4: Attempt to pull the drawer to check if it is open
        # This will help us discover the (is-open drawer1) predicate
        try:
            print("[Exploration] Attempting to pull drawer1 to check if it is open.")
            obs, reward, done = pull(
                env,
                task,
                gripper_name,
                drawer_name
            )
            if done:
                print("[Exploration] Task ended during pull!")
                return
        except Exception as e:
            print(f"[Error] Exception during pull: {e}")
            shutdown_environment(env)
            return

        print("[Exploration] Exploration phase complete. (is-open drawer1) predicate should now be known.")

        # === Main Task Plan (After Exploration) ===
        # At this point, you can proceed with the rest of the oracle plan, e.g., pick/place objects, etc.
        # For this code, we focus on the exploration phase as per the feedback.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()