# run_skeleton_task.py (Completed with Exploration Phase and Skill Calls)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback and exploration knowledge suggest that some predicates (e.g., lock-known, identified, temperature-known, etc.)
        # may be missing or not properly handled in the domain/problem.
        # We will perform an exploration phase to check which predicates are missing or not being updated as expected.

        # For demonstration, we will attempt to use the available skills to interact with all objects and locations,
        # and print out any issues or missing information encountered.

        # Get the list of all objects and locations from positions
        object_names = [name for name in positions if not name.startswith('location_')]
        location_names = [name for name in positions if name.startswith('location_')]

        # If location names are not explicitly given, try to infer from object positions
        if not location_names:
            # Fallback: treat all unique position tuples as locations
            unique_positions = set(tuple(pos) for pos in positions.values())
            location_names = [f"loc_{i}" for i, _ in enumerate(unique_positions)]

        # Print detected objects and locations
        print("[Exploration] Detected objects:", object_names)
        print("[Exploration] Detected locations:", location_names)

        # Try to pick and place each object at each location to see if any predicate is missing
        for obj in object_names:
            obj_pos = positions[obj]
            # Try to pick the object
            try:
                print(f"[Exploration] Attempting to pick object: {obj} at position {obj_pos}")
                obs, reward, done = pick(
                    env,
                    task,
                    target_pos=obj_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print(f"[Exploration] Task ended after picking {obj}!")
                    break
            except Exception as e:
                print(f"[Exploration] Exception during pick({obj}): {e}")

            # Try to place the object at its original position (or another location if available)
            try:
                place_pos = obj_pos
                print(f"[Exploration] Attempting to place object: {obj} at position {place_pos}")
                obs, reward, done = place(
                    env,
                    task,
                    target_pos=place_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print(f"[Exploration] Task ended after placing {obj}!")
                    break
            except Exception as e:
                print(f"[Exploration] Exception during place({obj}): {e}")

        # Try to rotate the gripper to a default orientation for each object (if applicable)
        for obj in object_names:
            try:
                # For demonstration, use a default quaternion (identity)
                target_quat = np.array([0, 0, 0, 1])
                print(f"[Exploration] Attempting to rotate gripper for object: {obj}")
                obs, reward, done = rotate(
                    env,
                    task,
                    target_quat=target_quat,
                    max_steps=50,
                    threshold=0.05,
                    timeout=5.0
                )
                if done:
                    print(f"[Exploration] Task ended after rotating for {obj}!")
                    break
            except Exception as e:
                print(f"[Exploration] Exception during rotate({obj}): {e}")

        # Try to pull each object (if applicable)
        for obj in object_names:
            try:
                print(f"[Exploration] Attempting to pull object: {obj}")
                obs, reward, done = pull(
                    env,
                    task,
                    obj
                )
                if done:
                    print(f"[Exploration] Task ended after pulling {obj}!")
                    break
            except Exception as e:
                print(f"[Exploration] Exception during pull({obj}): {e}")

        # Try to move to each location (if move skill is implemented and available)
        # Since the move skill signature is not specified, this is a placeholder
        # Uncomment and adapt if move(env, task, from_loc, to_loc) is available
        # for from_loc in location_names:
        #     for to_loc in location_names:
        #         if from_loc != to_loc:
        #             try:
        #                 print(f"[Exploration] Attempting to move from {from_loc} to {to_loc}")
        #                 obs, reward, done = move(
        #                     env,
        #                     task,
        #                     from_loc,
        #                     to_loc
        #                 )
        #                 if done:
        #                     print(f"[Exploration] Task ended after move from {from_loc} to {to_loc}!")
        #                     break
        #             except Exception as e:
        #                 print(f"[Exploration] Exception during move({from_loc}, {to_loc}): {e}")

        print("[Exploration] Exploration phase complete. Check above for any missing predicates or errors.")

        # === End of Exploration Phase ===

        # TODO: After exploration, use the information to update your plan or report missing predicates.
        # For now, the code only explores and prints out issues.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
