# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'object_1': (0,0,0), 'object_2': (1,1,1), ...}
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (at anchor-pos-bottom bottom)
        # The missing predicate is likely related to the anchor position of the drawer (e.g., is-anchor-pos).
        # We need to explore and check if the anchor position is correctly set for the drawer.

        # Let's print out all positions to help with exploration
        print("[Exploration] Object positions in the environment:")
        for name, pos in positions.items():
            print(f"    {name}: {pos}")

        # Try to find anchor positions and check if the predicate is missing
        anchor_pos_name = None
        for name in positions:
            if "anchor" in name or "bottom" in name:
                anchor_pos_name = name
                break

        if anchor_pos_name is None:
            print("[Exploration] No anchor position found in object positions. The predicate (is-anchor-pos ...) may be missing.")
        else:
            print(f"[Exploration] Anchor position candidate: {anchor_pos_name} at {positions[anchor_pos_name]}")
            # In a real exploration, we might try to move the gripper to this position and see if the drawer can be manipulated.

        # === Main Task Plan Execution ===
        # For demonstration, let's attempt to pick an object at the anchor position if available.
        # Use only the available skills: ['pick', 'place', 'move', 'rotate', 'pull']

        # Example: Try to pick an object at the anchor position (if any)
        try:
            if anchor_pos_name is not None:
                target_pos = positions[anchor_pos_name]
                print(f"[Task] Attempting to pick at anchor position: {anchor_pos_name} ({target_pos})")
                obs, reward, done = pick(
                    env,
                    task,
                    target_pos=target_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print("[Task] Task ended after pick at anchor position!")
                    return
            else:
                # Fallback: Try to pick any object
                for obj_name, obj_pos in positions.items():
                    print(f"[Task] Attempting to pick object: {obj_name} at {obj_pos}")
                    try:
                        obs, reward, done = pick(
                            env,
                            task,
                            target_pos=obj_pos,
                            approach_distance=0.15,
                            max_steps=100,
                            threshold=0.01,
                            approach_axis='z',
                            timeout=10.0
                        )
                        if done:
                            print(f"[Task] Task ended after pick of {obj_name}!")
                            return
                    except Exception as e:
                        print(f"[Task] Failed to pick {obj_name}: {e}")
        except Exception as e:
            print(f"[Task] Exception during pick: {e}")

        # Additional exploration: Try to rotate or pull if possible
        # (Assume we have a drawer and a gripper in the positions dictionary)
        drawer_name = None
        gripper_name = None
        for name in positions:
            if "drawer" in name:
                drawer_name = name
            if "gripper" in name:
                gripper_name = name

        if drawer_name and gripper_name:
            print(f"[Task] Attempting to rotate gripper {gripper_name} for drawer {drawer_name}")
            try:
                # For demonstration, use dummy angles 'zero_deg' and 'ninety_deg'
                obs, reward, done = rotate(
                    env,
                    task,
                    gripper_name,
                    'zero_deg',
                    'ninety_deg'
                )
                if done:
                    print("[Task] Task ended after rotate!")
                    return
            except Exception as e:
                print(f"[Task] Failed to rotate: {e}")

            print(f"[Task] Attempting to pull drawer {drawer_name} with gripper {gripper_name}")
            try:
                obs, reward, done = pull(
                    env,
                    task,
                    gripper_name,
                    drawer_name
                )
                if done:
                    print("[Task] Task ended after pull!")
                    return
            except Exception as e:
                print(f"[Task] Failed to pull: {e}")

        print("[Task] Exploration and main plan execution completed.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()