# run_skeleton_task.py (Completed with Exploration Phase and Skill Calls)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        print("[Info] Object positions:", positions)

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback and exploration PDDL suggest that some predicates (e.g., lock-known, durability-known, etc.)
        # may be missing or not directly observable. We will perform exploration actions to infer these.

        # For demonstration, we will attempt to:
        # 1. Move to each location to identify objects and their properties.
        # 2. Pick up objects to learn their weight/durability.
        # 3. Pull on objects (e.g., drawers) to learn lock state.

        # The available skills are: ['pick', 'place', 'move', 'rotate', 'pull']

        # --- Example: Exploration for lock-known predicate on drawers ---
        # We'll iterate over all objects and try to pull drawers to see if we can learn their lock state.

        # For this example, let's assume positions contains keys like 'drawer1', 'object1', etc.
        # and each value is a dict with at least 'type' and 'position' fields.

        # We'll keep track of which predicates we have explored.
        explored_predicates = set()

        for obj_name, obj_info in positions.items():
            obj_type = obj_info.get('type', '')
            obj_pos = obj_info.get('position', None)
            if obj_pos is None:
                print(f"[Warning] No position for {obj_name}, skipping.")
                continue

            # --- 1. Move to the object's location to identify it ---
            try:
                print(f"[Exploration] Moving to {obj_name} at {obj_pos} to identify.")
                # The move skill may require from_pos and to_pos; here we use current gripper pos as from_pos.
                current_obs = task.get_observation()
                gripper_pos = getattr(current_obs, 'gripper_pose', None)
                if gripper_pos is not None:
                    from_pos = gripper_pos[:3]
                else:
                    from_pos = np.zeros(3)
                # Call move skill (if available), else just print
                if 'move' in globals():
                    obs, reward, done = move(
                        env,
                        task,
                        from_pos=from_pos,
                        to_pos=obj_pos,
                        max_steps=100,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=10.0
                    )
                else:
                    print("[Exploration] No move skill implemented.")
                explored_predicates.add('identified')
            except Exception as e:
                print(f"[Error] Move to {obj_name} failed: {e}")

            # --- 2. If object is pickable, pick it to learn weight/durability ---
            if obj_type in ['object', 'item', 'ball', 'can', 'bottle']:  # Example pickable types
                try:
                    print(f"[Exploration] Attempting to pick {obj_name} at {obj_pos}.")
                    if 'pick' in globals():
                        obs, reward, done = pick(
                            env,
                            task,
                            target_pos=obj_pos,
                            approach_distance=0.15,
                            max_steps=100,
                            threshold=0.01,
                            approach_axis='z',
                            timeout=10.0
                        )
                        explored_predicates.add('weight-known')
                        explored_predicates.add('durability-known')
                        # Place it back if possible
                        if 'place' in globals():
                            print(f"[Exploration] Placing {obj_name} back at {obj_pos}.")
                            obs, reward, done = place(
                                env,
                                task,
                                target_pos=obj_pos,
                                approach_distance=0.15,
                                max_steps=100,
                                threshold=0.01,
                                approach_axis='z',
                                timeout=10.0
                            )
                    else:
                        print("[Exploration] No pick skill implemented.")
                except Exception as e:
                    print(f"[Error] Pick/place for {obj_name} failed: {e}")

            # --- 3. If object is a drawer, try to pull to learn lock state ---
            if obj_type == 'drawer':
                try:
                    print(f"[Exploration] Attempting to pull {obj_name} at {obj_pos} to check lock state.")
                    # For pull, we need to specify a direction and distance.
                    # We'll use a small pull along the x-axis as an example.
                    pull_axis = 'x'
                    pull_distance = 0.05  # 5cm
                    # Validate pull_axis
                    valid_axes = ['x', '-x', 'y', '-y', 'z', '-z']
                    if pull_axis not in valid_axes:
                        print(f"[Error] Invalid pull_axis: {pull_axis}")
                        continue
                    # Check for gripper_pose attribute and shape
                    current_obs = task.get_observation()
                    if not hasattr(current_obs, 'gripper_pose') or current_obs.gripper_pose is None:
                        print("[Error] gripper_pose attribute missing in observation.")
                        continue
                    if len(current_obs.gripper_pose) < 7:
                        print("[Error] gripper_pose does not have enough elements.")
                        continue
                    # Call pull skill
                    obs, reward, done = pull(
                        env,
                        task,
                        pull_distance=pull_distance,
                        pull_axis=pull_axis,
                        max_steps=100,
                        threshold=0.01,
                        timeout=10.0
                    )
                    explored_predicates.add('lock-known')
                except Exception as e:
                    print(f"[Error] Pull for {obj_name} failed: {e}")

        print("[Exploration] Explored predicates:", explored_predicates)
        print("[Exploration] Exploration phase complete. If any predicate is still missing, further exploration may be needed.")

        # === Main Task Plan Execution ===
        # After exploration, proceed with the main plan using the available skills.
        # For demonstration, we show a generic plan for picking and placing an object.

        # Example: Pick up the first pickable object and place it at a target location (if any)
        pickable_obj = None
        for obj_name, obj_info in positions.items():
            if obj_info.get('type', '') in ['object', 'item', 'ball', 'can', 'bottle']:
                pickable_obj = (obj_name, obj_info)
                break

        if pickable_obj is not None:
            obj_name, obj_info = pickable_obj
            obj_pos = obj_info.get('position', None)
            if obj_pos is not None:
                try:
                    print(f"[Task] Picking {obj_name} at {obj_pos}.")
                    obs, reward, done = pick(
                        env,
                        task,
                        target_pos=obj_pos,
                        approach_distance=0.15,
                        max_steps=100,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=10.0
                    )
                    if done:
                        print("[Task] Task ended after pick!")
                        return
                    # Place at a new location (for demonstration, offset by +0.2m in x)
                    place_pos = np.array(obj_pos) + np.array([0.2, 0, 0])
                    print(f"[Task] Placing {obj_name} at {place_pos}.")
                    obs, reward, done = place(
                        env,
                        task,
                        target_pos=place_pos,
                        approach_distance=0.15,
                        max_steps=100,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=10.0
                    )
                    if done:
                        print("[Task] Task ended after place!")
                        return
                except Exception as e:
                    print(f"[Error] Task pick/place failed: {e}")
        else:
            print("[Task] No pickable object found for demonstration.")

        # You can add more plan steps here using the available skills as needed.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()
