# run_skeleton_task.py (Completed with Exploration Phase and Feedback Handling)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Based on feedback and exploration knowledge, we need to check for missing predicates
        # such as 'identified', 'temperature-known', 'weight-known', 'durability-known', 'lock-known'.
        # We'll perform exploration actions to discover which predicate is missing.

        # For demonstration, let's assume we have a list of objects and locations from positions
        # and that the robot is named 'robot1' (if needed).
        # We'll try to move to each location and pick each object, checking for state changes.

        # Extract object and location names from positions
        object_names = []
        location_names = set()
        for name, pos in positions.items():
            if name.startswith('object') or name.startswith('rubbish') or name.startswith('drawer'):
                object_names.append(name)
            if isinstance(pos, dict) and 'location' in pos:
                location_names.add(pos['location'])
            elif isinstance(pos, (tuple, list)) and len(pos) == 3:
                # Assume the key is the location name if not object
                pass

        # Remove forced object issue: ensure 'rubbish' is in object_names if present in positions
        if 'rubbish' in positions and 'rubbish' not in object_names:
            object_names.append('rubbish')

        # Safety: Remove duplicates
        object_names = list(set(object_names))

        # For the exploration, we will:
        # 1. Move to each location (if move skill is available)
        # 2. Try to pick each object at its location (if pick skill is available)
        # 3. After each action, check for new state information (e.g., print obs or log)

        # For this code, we assume the robot starts at some initial location.
        # We'll use the available skills: ['pick', 'place', 'move', 'rotate', 'pull']

        # --- Exploration: Move to each object's location and try to pick ---
        for obj in object_names:
            try:
                # Get the object's position and (if available) its location
                obj_pos = positions.get(obj, None)
                if obj_pos is None:
                    print(f"[Exploration] Warning: No position found for object '{obj}'. Skipping.")
                    continue

                # If the object has a location, use it; else, try to infer
                if isinstance(obj_pos, dict) and 'location' in obj_pos:
                    obj_location = obj_pos['location']
                else:
                    # Fallback: use object name as location if possible
                    obj_location = None

                # 1. Move to the object's location (if move skill is available and location known)
                if obj_location is not None and 'move' in globals():
                    print(f"[Exploration] Moving to location of '{obj}': {obj_location}")
                    try:
                        obs, reward, done = move(env, task, obj_location)
                        if done:
                            print(f"[Exploration] Task ended after move to {obj_location}!")
                            return
                    except Exception as e:
                        print(f"[Exploration] Exception during move to {obj_location}: {e}")

                # 2. Try to pick the object (if pick skill is available)
                if 'pick' in globals():
                    print(f"[Exploration] Attempting to pick '{obj}' at location '{obj_location}'")
                    try:
                        obs, reward, done = pick(env, task, obj, obj_location)
                        if done:
                            print(f"[Exploration] Task ended after picking {obj}!")
                            return
                    except Exception as e:
                        print(f"[Exploration] Exception during pick of {obj}: {e}")

                # 3. After pick, try to place the object back (if place skill is available)
                if 'place' in globals() and obj_location is not None:
                    print(f"[Exploration] Attempting to place '{obj}' back at location '{obj_location}'")
                    try:
                        obs, reward, done = place(env, task, obj, obj_location)
                        if done:
                            print(f"[Exploration] Task ended after placing {obj}!")
                            return
                    except Exception as e:
                        print(f"[Exploration] Exception during place of {obj}: {e}")

                # 4. Try to pull (if pull skill is available and object is a drawer)
                if 'pull' in globals() and 'drawer' in obj:
                    print(f"[Exploration] Attempting to pull '{obj}'")
                    try:
                        obs, reward, done = pull(env, task, obj)
                        if done:
                            print(f"[Exploration] Task ended after pulling {obj}!")
                            return
                    except Exception as e:
                        print(f"[Exploration] Exception during pull of {obj}: {e}")

                # 5. Try to rotate (if rotate skill is available and object is a gripper or drawer)
                if 'rotate' in globals() and ('gripper' in obj or 'drawer' in obj):
                    print(f"[Exploration] Attempting to rotate '{obj}'")
                    try:
                        # For demonstration, use a default target quaternion (identity)
                        target_quat = np.array([0, 0, 0, 1])
                        obs, reward, done = rotate(env, task, target_quat)
                        if done:
                            print(f"[Exploration] Task ended after rotating {obj}!")
                            return
                    except Exception as e:
                        print(f"[Exploration] Exception during rotate of {obj}: {e}")

                # 6. After each action, print the observation for analysis
                print(f"[Exploration] Observation after actions on '{obj}': {obs}")

            except Exception as e:
                print(f"[Exploration] General exception for object '{obj}': {e}")

        # === End of Exploration Phase ===

        # At this point, based on the observations and feedback, you can analyze which predicates
        # (e.g., identified, temperature-known, weight-known, durability-known, lock-known)
        # are missing or need to be further explored.

        print("[Exploration] Completed exploration of all objects.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
