# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'gripper1': ..., 'drawer1': ..., 'angle_ninety': ..., ...}
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (rotated gripper1 angle_ninety) is missing in the initial state.
        # We need to explore and set the gripper to the correct rotation before proceeding.

        # 1. Check if gripper1 is at angle_ninety; if not, rotate it.
        gripper_name = 'gripper1'
        target_angle = 'angle_ninety'
        try:
            gripper_pos = positions[gripper_name]
        except KeyError:
            print(f"[Error] Gripper '{gripper_name}' not found in positions.")
            shutdown_environment(env)
            return

        # For demonstration, assume we can get the current angle from observation or positions
        # If not, always perform the rotate action to ensure the predicate is satisfied.
        print(f"[Exploration] Ensuring {gripper_name} is rotated to {target_angle}.")

        # The rotate skill expects: (env, task, gripper, from_angle, to_angle)
        # We'll try to get the current angle, else default to rotating from 'angle_zero' to 'angle_ninety'
        current_angle = None
        if 'angle_ninety' in positions:
            to_angle = 'angle_ninety'
        else:
            print("[Warning] 'angle_ninety' not found in positions. Using default value.")
            to_angle = 'angle_ninety'
        # Try to infer current angle; if not available, use 'angle_zero'
        if 'angle_zero' in positions:
            from_angle = 'angle_zero'
        else:
            from_angle = 'angle_zero'  # Default fallback

        try:
            obs, reward, done = rotate(env, task, gripper_name, from_angle, to_angle)
            print(f"[Exploration] Rotated {gripper_name} from {from_angle} to {to_angle}.")
            if done:
                print("[Task] Task ended during rotation!")
                return
        except Exception as e:
            print(f"[Error] Exception during rotate: {e}")
            shutdown_environment(env)
            return

        # === Main Task Plan ===
        # At this point, the missing predicate (rotated gripper1 angle_ninety) is satisfied.
        # Continue with the rest of the plan using only available skills.

        # Example: Move gripper to a target position (if required)
        # For demonstration, let's move to the drawer handle position if available
        drawer_handle_key = 'drawer_handle'
        if drawer_handle_key in positions:
            target_pos = positions[drawer_handle_key]
            print(f"[Task] Moving gripper to drawer handle at {target_pos}.")
            try:
                obs, reward, done = move(env, task, target_pos)
                if done:
                    print("[Task] Task ended during move!")
                    return
            except Exception as e:
                print(f"[Error] Exception during move: {e}")
                shutdown_environment(env)
                return
        else:
            print(f"[Warning] '{drawer_handle_key}' not found in positions. Skipping move.")

        # Example: Pull the drawer (if available)
        # Use the 'pull' skill if the drawer is present
        drawer_name = 'drawer1'
        if drawer_name in positions:
            try:
                obs, reward, done = pull(env, task, gripper_name, drawer_name)
                print(f"[Task] Pulled {drawer_name} using {gripper_name}.")
                if done:
                    print("[Task] Task ended after pull!")
                    return
            except Exception as e:
                print(f"[Error] Exception during pull: {e}")
                shutdown_environment(env)
                return
        else:
            print(f"[Warning] '{drawer_name}' not found in positions. Skipping pull.")

        # Example: Pick an object from inside the drawer (if available)
        object_key = 'object_1'
        if object_key in positions:
            object_pos = positions[object_key]
            print(f"[Task] Picking up {object_key} at {object_pos}.")
            try:
                obs, reward, done = pick(env, task, object_pos)
                if done:
                    print("[Task] Task ended after pick!")
                    return
            except Exception as e:
                print(f"[Error] Exception during pick: {e}")
                shutdown_environment(env)
                return
        else:
            print(f"[Warning] '{object_key}' not found in positions. Skipping pick.")

        # Example: Place the object at a target location (if available)
        place_target_key = 'place_target'
        if place_target_key in positions:
            place_pos = positions[place_target_key]
            print(f"[Task] Placing object at {place_pos}.")
            try:
                obs, reward, done = place(env, task, place_pos)
                if done:
                    print("[Task] Task ended after place!")
                    return
            except Exception as e:
                print(f"[Error] Exception during place: {e}")
                shutdown_environment(env)
                return
        else:
            print(f"[Warning] '{place_target_key}' not found in positions. Skipping place.")

        print("[Task] Task plan completed.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
