# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'rubbish': (x,y,z), 'trash_can': (x,y,z), ...}
        positions = get_object_positions()

        # --- Exploration Phase: Identify Missing Predicate ---
        # According to feedback and exploration knowledge, before picking or placing,
        # we need to ensure the object is identified (predicate: identified).
        # This is typically achieved by moving to the object's location.
        # We'll simulate this by moving to the rubbish location before picking.

        # --- Dynamic Parameter Selection (from feedback) ---
        # Use environment-specific values if available, else fallback to defaults.
        # For demonstration, we use the default values but show how to fetch them.
        approach_distance = positions.get('approach_distance', 0.15)
        max_steps = positions.get('max_steps', 100)
        threshold = positions.get('threshold', 0.01)
        approach_axis = positions.get('approach_axis', 'z')
        timeout = positions.get('timeout', 10.0)

        # --- Main Task Plan ---
        # 1. Move to rubbish location to "identify" it (exploration predicate)
        # 2. Pick up the rubbish
        # 3. Move to trash can location
        # 4. Place the rubbish in the trash can

        # --- Step 1: Move to rubbish location (to ensure 'identified' predicate) ---
        try:
            rubbish_pos = positions['rubbish']
        except KeyError:
            print("[Error] 'rubbish' object position not found in positions dictionary.")
            return

        print("[Task] Moving to rubbish at:", rubbish_pos)
        try:
            obs, reward, done = move(
                env,
                task,
                target_pos=rubbish_pos,
                approach_distance=approach_distance,
                max_steps=max_steps,
                threshold=threshold,
                approach_axis=approach_axis,
                timeout=timeout
            )
        except Exception as e:
            print(f"[Error] Exception during move to rubbish: {e}")
            return

        if done:
            print("[Task] Task ended during move to rubbish!")
            return

        # --- Step 2: Pick up the rubbish ---
        print("[Task] Picking up rubbish at:", rubbish_pos)
        try:
            obs, reward, done = pick(
                env,
                task,
                target_pos=rubbish_pos,
                approach_distance=approach_distance,
                max_steps=max_steps,
                threshold=threshold,
                approach_axis=approach_axis,
                timeout=timeout
            )
        except Exception as e:
            print(f"[Error] Exception during pick: {e}")
            return

        if done:
            print("[Task] Task ended during pick!")
            return

        # --- Step 3: Move to trash can location ---
        try:
            trash_can_pos = positions['trash_can']
        except KeyError:
            print("[Error] 'trash_can' object position not found in positions dictionary.")
            return

        print("[Task] Moving to trash can at:", trash_can_pos)
        try:
            obs, reward, done = move(
                env,
                task,
                target_pos=trash_can_pos,
                approach_distance=approach_distance,
                max_steps=max_steps,
                threshold=threshold,
                approach_axis=approach_axis,
                timeout=timeout
            )
        except Exception as e:
            print(f"[Error] Exception during move to trash can: {e}")
            return

        if done:
            print("[Task] Task ended during move to trash can!")
            return

        # --- Step 4: Place the rubbish in the trash can ---
        print("[Task] Placing rubbish in trash can at:", trash_can_pos)
        try:
            # Before placing, check position and orientation as per feedback
            obs = task.get_observation()
            gripper_pos = obs.gripper_pose[:3]
            gripper_quat = obs.gripper_pose[3:7]
            # Check position
            dist_to_target = np.linalg.norm(gripper_pos - trash_can_pos)
            if dist_to_target > threshold:
                print(f"[Warning] Gripper not at target position for place (distance: {dist_to_target:.4f}). Moving closer.")
                obs, reward, done = move(
                    env,
                    task,
                    target_pos=trash_can_pos,
                    approach_distance=0.0,
                    max_steps=max_steps,
                    threshold=threshold,
                    approach_axis=approach_axis,
                    timeout=timeout
                )
                if done:
                    print("[Task] Task ended during final approach for place!")
                    return
            # Check orientation (optional, as per feedback)
            # For demonstration, assume desired orientation is same as current
            # In practice, compare with desired_quat if available

            # Collision detection (feedback): Not implemented, but could be added here

            obs, reward, done = place(
                env,
                task,
                target_pos=trash_can_pos,
                approach_distance=approach_distance,
                max_steps=max_steps,
                threshold=threshold,
                approach_axis=approach_axis,
                timeout=timeout
            )
        except Exception as e:
            print(f"[Error] Exception during place: {e}")
            return

        if done:
            print("[Task] Task ended after place!")
            return

        print("[Task] Successfully placed rubbish in trash can.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()