# run_skeleton_task.py (Completed with Exploration Phase and Feedback Handling)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def euler_from_quat(quat):
    """
    Converts quaternion (x, y, z, w) to Euler angles (roll, pitch, yaw).
    """
    x, y, z, w = quat
    t0 = +2.0 * (w * x + y * z)
    t1 = +1.0 - 2.0 * (x * x + y * y)
    roll_x = np.arctan2(t0, t1)

    t2 = +2.0 * (w * y - z * x)
    t2 = np.clip(t2, -1.0, 1.0)
    pitch_y = np.arcsin(t2)

    t3 = +2.0 * (w * z + x * y)
    t4 = +1.0 - 2.0 * (y * y + z * z)
    yaw_z = np.arctan2(t3, t4)

    return np.array([roll_x, pitch_y, yaw_z])

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration and feedback handling.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate/Knowledge ===
        # The feedback and exploration domain suggest that the robot may need to discover some property (e.g., lock-known, weight-known, durability-known, temperature-known, identified)
        # We'll attempt to use available skills to "probe" the environment and check for missing knowledge.

        # For demonstration, let's try to pick and pull on all objects that look like drawers or handles, and check for lock state.
        # We'll also check for weight/durability by picking up objects.

        # This phase is generic: in a real scenario, you'd parse the PDDL and initial state to determine which predicates are missing.
        # Here, we simulate this by iterating over objects and using available skills.

        # Safety: We'll wrap all skill calls in try/except to avoid crashes.

        # 1. Try to pick up all objects to check for weight/durability knowledge
        for obj_name, obj_info in positions.items():
            obj_type = obj_info.get('type', '')
            obj_pos = obj_info.get('position', None)
            obj_loc = obj_info.get('location', None)
            if obj_type in ['object', 'drawer', 'handle'] and obj_pos is not None:
                print(f"[Exploration] Attempting to pick {obj_name} at {obj_pos} ...")
                try:
                    obs, reward, done = pick(
                        env,
                        task,
                        target_pos=obj_pos,
                        approach_distance=0.15,
                        max_steps=100,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=10.0
                    )
                    if done:
                        print(f"[Exploration] Task ended after picking {obj_name}!")
                        return
                except Exception as e:
                    print(f"[Exploration] Could not pick {obj_name}: {e}")

        # 2. Try to pull on all drawers/handles to check for lock-known predicate
        for obj_name, obj_info in positions.items():
            obj_type = obj_info.get('type', '')
            obj_pos = obj_info.get('position', None)
            if obj_type in ['drawer', 'handle'] and obj_pos is not None:
                print(f"[Exploration] Attempting to pull {obj_name} at {obj_pos} ...")
                try:
                    obs, reward, done = pull(
                        env,
                        task,
                        target_pos=obj_pos,
                        approach_distance=0.10,
                        max_steps=80,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=8.0
                    )
                    if done:
                        print(f"[Exploration] Task ended after pulling {obj_name}!")
                        return
                except Exception as e:
                    print(f"[Exploration] Could not pull {obj_name}: {e}")

        # 3. Try to rotate gripper at various positions to check for angle/rotated predicates
        for obj_name, obj_info in positions.items():
            obj_type = obj_info.get('type', '')
            obj_pos = obj_info.get('position', None)
            if obj_type in ['drawer', 'handle'] and obj_pos is not None:
                # Try rotating to 90 degrees (in quaternion)
                print(f"[Exploration] Attempting to rotate gripper at {obj_name} ...")
                try:
                    # Example: rotate to 90 degrees about z axis
                    angle_rad = np.pi / 2
                    target_quat = np.array([0, 0, np.sin(angle_rad/2), np.cos(angle_rad/2)])  # z-axis rotation
                    obs, reward, done = rotate(
                        env,
                        task,
                        target_quat=target_quat,
                        max_steps=60,
                        threshold=0.05,
                        timeout=6.0
                    )
                    if done:
                        print(f"[Exploration] Task ended after rotating at {obj_name}!")
                        return
                except Exception as e:
                    print(f"[Exploration] Could not rotate at {obj_name}: {e}")

        # 4. Try to place objects at known locations (if any) to check for placeability
        for obj_name, obj_info in positions.items():
            obj_type = obj_info.get('type', '')
            obj_pos = obj_info.get('position', None)
            obj_loc = obj_info.get('location', None)
            if obj_type == 'object' and obj_pos is not None and obj_loc is not None:
                print(f"[Exploration] Attempting to place {obj_name} at {obj_loc} ...")
                try:
                    obs, reward, done = place(
                        env,
                        task,
                        target_pos=obj_pos,
                        approach_distance=0.12,
                        max_steps=80,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=8.0
                    )
                    if done:
                        print(f"[Exploration] Task ended after placing {obj_name}!")
                        return
                except Exception as e:
                    print(f"[Exploration] Could not place {obj_name}: {e}")

        # 5. Move to all known locations to check for identified/temperature-known predicates
        for obj_name, obj_info in positions.items():
            obj_loc = obj_info.get('location', None)
            if obj_loc is not None:
                print(f"[Exploration] Attempting to move to location {obj_loc} ...")
                try:
                    obs, reward, done = move(
                        env,
                        task,
                        target_location=obj_loc,
                        approach_distance=0.20,
                        max_steps=60,
                        threshold=0.01,
                        timeout=8.0
                    )
                    if done:
                        print(f"[Exploration] Task ended after moving to {obj_loc}!")
                        return
                except Exception as e:
                    print(f"[Exploration] Could not move to {obj_loc}: {e}")

        print("[Exploration] Exploration phase complete. If any missing predicate was discovered, it should now be known.")

        # === Main Task Plan (Oracle Plan Execution) ===
        # At this point, you would execute the oracle plan step-by-step using the available skills.
        # For each step, use the corresponding skill function (pick, place, move, rotate, pull) with the correct arguments.
        # Example:
        # obs, reward, done = pick(env, task, ...)
        # if done: return

        # (You would fill in the actual plan here, using the same try/except safety as above.)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()