# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'object_1': (x,y,z), ...}
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback indicates (handempty) predicate is missing or not handled.
        # We will explore the environment to check if the robot's hand is empty before picking.
        # This is crucial for correct pick/place logic.

        # For demonstration, let's assume we want to pick up 'object_1' and place it at 'target_location'.
        # These names should match your environment's object keys.
        object_key = None
        target_location_key = None

        # Try to find a pickable object and a target location from positions
        for key in positions:
            if 'object' in key or 'obj' in key:
                object_key = key
            if 'target' in key or 'goal' in key or 'bin' in key or 'place' in key or 'location' in key:
                target_location_key = key
        # Fallback: just pick the first two keys if not found
        if object_key is None and len(positions) > 0:
            object_key = list(positions.keys())[0]
        if target_location_key is None and len(positions) > 1:
            target_location_key = list(positions.keys())[1]

        if object_key is None or target_location_key is None:
            print("[Exploration] Could not find suitable object or target location in positions:", positions)
            return

        object_pos = positions[object_key]
        target_pos = positions[target_location_key]

        print(f"[Exploration] Selected object: {object_key} at {object_pos}")
        print(f"[Exploration] Selected target: {target_location_key} at {target_pos}")

        # === Check for 'handempty' predicate before pick ===
        # We will try to pick only if the robot's hand is empty.
        # If not, we will print a warning and skip pick.

        obs = task.get_observation()
        # The observation may or may not have a 'handempty' field; check for it.
        handempty = None
        if hasattr(obs, 'handempty'):
            handempty = obs.handempty
        elif isinstance(obs, dict) and 'handempty' in obs:
            handempty = obs['handempty']
        else:
            # Try to infer from gripper state (e.g., gripper_opened or holding)
            if hasattr(obs, 'gripper_opened'):
                handempty = obs.gripper_opened > 0.9
            elif 'gripper_opened' in obs:
                handempty = obs['gripper_opened'] > 0.9
            else:
                # If not available, assume hand is empty at start
                handempty = True

        print(f"[Exploration] handempty: {handempty}")

        # If not handempty, we cannot pick; print feedback
        if not handempty:
            print("[Exploration] Robot hand is not empty! Cannot pick. Exploration reveals missing (handempty) predicate handling.")
            # Optionally, try to place or drop to empty the hand
            # For now, abort
            return

        # === Execute Oracle Plan: pick and place ===
        # 1. Pick the object
        try:
            print(f"[Task] Attempting to pick {object_key} at {object_pos}")
            obs, reward, done = pick(
                env,
                task,
                target_pos=object_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Task] Task ended after pick!")
                return
        except Exception as e:
            print(f"[Task] Exception during pick: {e}")
            return

        # 2. Place the object at the target location
        try:
            print(f"[Task] Attempting to place {object_key} at {target_pos}")
            obs, reward, done = place(
                env,
                task,
                target_pos=target_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Task] Task ended after place!")
                return
        except Exception as e:
            print(f"[Task] Exception during place: {e}")
            return

        print("[Task] Pick and place completed successfully.")

        # === Additional Exploration: Try other skills if needed ===
        # For completeness, you could try move, rotate, pull, etc., if your plan requires.
        # For now, we focus on pick and place as per the feedback and available skills.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()