import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use provided skills: pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Task: Pull open any unlocked drawer, then drop the 2 tomatoes onto the plate.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Object names from object list: 'item1', 'item2', 'item3', 'plate', etc.
        # Assume 'item1' and 'item2' are the tomatoes, 'plate' is the destination.

        # Defensive: Check for required objects
        required_objects = ['item1', 'item2', 'plate']
        for obj in required_objects:
            if obj not in positions:
                raise RuntimeError(f"Required object '{obj}' not found in object_positions.")

        tomato1_pos = positions['item1']
        tomato2_pos = positions['item2']
        plate_pos = positions['plate']

        # === Step 1: Pull open any unlocked drawer ===
        # We'll choose the 'bottom' drawer (as an example), and try to pull it open.
        # To pull, we must:
        #   - Move gripper to the anchor position of the drawer
        #   - Rotate gripper if needed
        #   - (Optionally) pick-drawer (not implemented as a skill, so we just position)
        #   - Pull along the drawer axis

        # For 'bottom' drawer, anchor position is 'bottom_anchor_pos'
        # We'll use the position from object_positions if available, else fallback to a default
        drawer_anchor_key = 'bottom_anchor_pos'
        if drawer_anchor_key not in positions:
            raise RuntimeError(f"Drawer anchor position '{drawer_anchor_key}' not found in object_positions.")
        drawer_anchor_pos = positions[drawer_anchor_key]

        # 1. Move gripper to anchor position above the drawer (approach from above)
        print("[Task] Moving gripper above drawer anchor position...")
        approach_offset = np.array([0, 0, 0.10])  # 10cm above anchor
        drawer_approach_pos = drawer_anchor_pos + approach_offset
        obs, reward, done = move(
            env, task, target_pos=drawer_approach_pos, max_steps=100, threshold=0.01, timeout=10.0
        )
        if done:
            print("[Task] Task ended during move to drawer anchor approach!")
            return

        # 2. Move down to anchor position
        print("[Task] Moving gripper to drawer anchor position...")
        obs, reward, done = move(
            env, task, target_pos=drawer_anchor_pos, max_steps=100, threshold=0.01, timeout=10.0
        )
        if done:
            print("[Task] Task ended during move to drawer anchor!")
            return

        # 3. (Optional) Rotate gripper to align with drawer axis (if needed)
        # For simplicity, we skip explicit rotation unless required by the environment.

        # 4. Close gripper to "grasp" the drawer handle (simulate pick-drawer)
        # Use the pick skill to close gripper at the anchor position
        print("[Task] Grasping drawer handle (simulated pick)...")
        obs, reward, done = pick(
            env, task, target_pos=drawer_anchor_pos, approach_distance=0.0, max_steps=50, threshold=0.01, approach_axis='z', timeout=5.0
        )
        if done:
            print("[Task] Task ended during drawer handle grasp!")
            return

        # 5. Pull the drawer open along the x-axis (assume positive x is pull direction)
        print("[Task] Pulling the drawer open...")
        pull_distance = 0.15  # 15cm pull
        obs, reward, done = pull(
            env, task, pull_distance=pull_distance, pull_axis='x', max_steps=100, threshold=0.01, timeout=10.0
        )
        if done:
            print("[Task] Task ended during drawer pull!")
            return

        # 6. (Optional) Open gripper to release drawer handle
        # Use place skill at current position to open gripper
        print("[Task] Releasing drawer handle (simulated place)...")
        obs, reward, done = place(
            env, task, target_pos=drawer_anchor_pos, approach_distance=0.0, max_steps=50, threshold=0.01, approach_axis='z', timeout=5.0
        )
        if done:
            print("[Task] Task ended during drawer handle release!")
            return

        # === Step 2: Pick up first tomato and place on plate ===
        print("[Task] Picking up first tomato (item1)...")
        obs, reward, done = pick(
            env, task, target_pos=tomato1_pos, approach_distance=0.15, max_steps=100, threshold=0.01, approach_axis='z', timeout=10.0
        )
        if done:
            print("[Task] Task ended during pick of item1!")
            return

        print("[Task] Placing first tomato (item1) on plate...")
        obs, reward, done = place(
            env, task, target_pos=plate_pos, approach_distance=0.15, max_steps=100, threshold=0.01, approach_axis='z', timeout=10.0
        )
        if done:
            print("[Task] Task ended during place of item1!")
            return

        # === Step 3: Pick up second tomato and place on plate ===
        print("[Task] Picking up second tomato (item2)...")
        obs, reward, done = pick(
            env, task, target_pos=tomato2_pos, approach_distance=0.15, max_steps=100, threshold=0.01, approach_axis='z', timeout=10.0
        )
        if done:
            print("[Task] Task ended during pick of item2!")
            return

        print("[Task] Placing second tomato (item2) on plate...")
        obs, reward, done = place(
            env, task, target_pos=plate_pos, approach_distance=0.15, max_steps=100, threshold=0.01, approach_axis='z', timeout=10.0
        )
        if done:
            print("[Task] Task ended during place of item2!")
            return

        print("[Task] Task completed successfully! Reward:", reward)

    except Exception as e:
        print(f"[Task] Exception occurred: {e}")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()