# run_skeleton_task.py

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use provided skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Object names from object list:
        # [
        # 'bottom_anchor_pos', 'bottom_joint_pos', 'bottom_side_pos',
        # 'middle_anchor_pos', 'middle_joint_pos', 'middle_side_pos',
        # 'top_anchor_pos', 'top_joint_pos', 'top_side_pos',
        # 'item1', 'item2', 'item3', 'plate'
        # ]

        # For this task:
        # - Open an accessible drawer (to "unlock the cabinet")
        # - Move the 2 tomatoes onto the plate (leave other items undisturbed)
        # From the observation, the objects on the table are item1, item2, item3.
        # The two tomatoes are item1 and item2 (by convention; item3 is "rubbish").
        # The plate is "plate".
        # We'll use the "bottom" drawer (bottom_anchor_pos, bottom_side_pos) as the accessible drawer.

        # Defensive: check all required keys exist
        required_keys = [
            'bottom_side_pos', 'bottom_anchor_pos', 'item1', 'item2', 'plate'
        ]
        for k in required_keys:
            if k not in positions:
                raise RuntimeError(f"Required object position '{k}' not found in get_object_positions().")

        # Plan steps (from specification):
        # 1. move
        # 2. rotate
        # 3. move
        # 4. pull
        # 5. move
        # 6. pick
        # 7. place
        # 8. move
        # 9. pick
        # 10. place

        # Step 1: Move to the side of the bottom drawer (to approach for grasp)
        print("[Step 1] Move to bottom_side_pos (side of bottom drawer)")
        obs, reward, done = move(
            env,
            task,
            target_pos=positions['bottom_side_pos'],
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after move to bottom_side_pos!")
            return

        # Step 2: Rotate gripper to 90 degrees (for drawer grasp)
        print("[Step 2] Rotate gripper to 90 degrees (drawer grasp orientation)")
        # Assume 90 degrees about z axis from initial orientation
        # Get current orientation, then rotate about z by 90 deg
        from scipy.spatial.transform import Rotation as R
        current_quat = obs.gripper_pose[3:7]
        rot = R.from_quat(current_quat)
        rot_90 = R.from_euler('z', 90, degrees=True)
        target_rot = rot_90 * rot
        target_quat = target_rot.as_quat()
        obs, reward, done = rotate(
            env,
            task,
            target_quat=target_quat,
            max_steps=100,
            threshold=0.05,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after rotate!")
            return

        # Step 3: Move to the anchor position of the bottom drawer (for grasping handle)
        print("[Step 3] Move to bottom_anchor_pos (anchor/handle of bottom drawer)")
        obs, reward, done = move(
            env,
            task,
            target_pos=positions['bottom_anchor_pos'],
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after move to bottom_anchor_pos!")
            return

        # Step 4: Pull the drawer open (pull along -x axis, e.g., 0.15m)
        print("[Step 4] Pull the bottom drawer open")
        # Pull distance and axis may need tuning; assume -x is "outward"
        obs, reward, done = pull(
            env,
            task,
            pull_distance=0.15,
            pull_axis='-x',
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after pull!")
            return

        # Step 5: Move to item1 (first tomato) on the table
        print("[Step 5] Move to item1 (first tomato)")
        obs, reward, done = move(
            env,
            task,
            target_pos=positions['item1'],
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after move to item1!")
            return

        # Step 6: Pick item1
        print("[Step 6] Pick item1 (first tomato)")
        obs, reward, done = pick(
            env,
            task,
            target_pos=positions['item1'],
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after pick item1!")
            return

        # Step 7: Place item1 on the plate
        print("[Step 7] Place item1 on the plate")
        obs, reward, done = place(
            env,
            task,
            target_pos=positions['plate'],
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after place item1!")
            return

        # Step 8: Move to item2 (second tomato) on the table
        print("[Step 8] Move to item2 (second tomato)")
        obs, reward, done = move(
            env,
            task,
            target_pos=positions['item2'],
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after move to item2!")
            return

        # Step 9: Pick item2
        print("[Step 9] Pick item2 (second tomato)")
        obs, reward, done = pick(
            env,
            task,
            target_pos=positions['item2'],
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after pick item2!")
            return

        # Step 10: Place item2 on the plate
        print("[Step 10] Place item2 on the plate")
        obs, reward, done = place(
            env,
            task,
            target_pos=positions['plate'],
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after place item2!")
            return

        print("[Task] Task completed successfully! Reward:", reward)

    except Exception as e:
        print(f"[Task] Exception occurred: {e}")
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()