import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Task: Open one of the drawers in the cabinet, then put all tomatoes on the plate.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # --- Object name mapping (from object list and positions) ---
        # We'll use the 'middle' drawer for this plan.
        # Key objects: 'middle_anchor_pos', 'middle_side_pos', 'middle_joint_pos', 'tomato1', 'tomato2', 'plate'
        # The gripper starts at 'nowhere_pos' (not in object_positions, but initial obs), so we move to side, then anchor.

        # Defensive: check all required keys
        required_keys = [
            'middle_side_pos', 'middle_anchor_pos', 'middle_joint_pos',
            'tomato1', 'tomato2', 'plate'
        ]
        for k in required_keys:
            if k not in positions:
                raise RuntimeError(f"Missing key in object positions: {k}")

        # Drawer positions
        side_pos = positions['middle_side_pos']
        anchor_pos = positions['middle_anchor_pos']
        joint_pos = positions['middle_joint_pos']

        # Tomatoes and plate
        tomato1_pos = positions['tomato1']
        tomato2_pos = positions['tomato2']
        plate_pos = positions['plate']

        # === PLAN EXECUTION ===
        # Plan (from specification):
        # 1. rotate (to 90 deg for side approach)
        # 2. move (to side of drawer)
        # 3. move (to anchor of drawer)
        # 4. pick (drawer handle)
        # 5. pull (open drawer)
        # 6. pick (tomato1)
        # 7. place (tomato1 on plate)
        # 8. pick (tomato2)
        # 9. place (tomato2 on plate)

        # Step 1: rotate gripper to 90 deg (for side approach)
        # We'll use a quaternion for 90 deg rotation about z axis.
        # RLBench uses [x, y, z, w] (xyzw) convention.
        from scipy.spatial.transform import Rotation as R
        ninety_deg_quat = R.from_euler('z', 90, degrees=True).as_quat()  # [x, y, z, w]
        obs, reward, done = rotate(env, task, target_quat=ninety_deg_quat)
        if done:
            print("[Task] Task ended after rotate!")
            return

        # Step 2: move to side of middle drawer
        obs, reward, done = move(env, task, target_pos=side_pos)
        if done:
            print("[Task] Task ended after move to side!")
            return

        # Step 3: move to anchor of middle drawer (handle)
        obs, reward, done = move(env, task, target_pos=anchor_pos)
        if done:
            print("[Task] Task ended after move to anchor!")
            return

        # Step 4: pick the drawer handle (grasp at anchor)
        # Use approach_axis='y' to approach handle from the front (assuming handle is along y)
        obs, reward, done = pick(env, task, target_pos=anchor_pos, approach_distance=0.10, approach_axis='y')
        if done:
            print("[Task] Task ended after picking drawer handle!")
            return

        # Step 5: pull the drawer open
        # Pull along -y axis (assuming drawer opens toward -y), distance ~0.15m
        obs, reward, done = pull(env, task, pull_distance=0.15, pull_axis='-y')
        if done:
            print("[Task] Task ended after pulling drawer!")
            return

        # Step 6: pick tomato1 (from table)
        obs, reward, done = pick(env, task, target_pos=tomato1_pos, approach_distance=0.15, approach_axis='z')
        if done:
            print("[Task] Task ended after picking tomato1!")
            return

        # Step 7: place tomato1 on plate
        obs, reward, done = place(env, task, target_pos=plate_pos, approach_distance=0.15, approach_axis='z')
        if done:
            print("[Task] Task ended after placing tomato1!")
            return

        # Step 8: pick tomato2 (from table)
        obs, reward, done = pick(env, task, target_pos=tomato2_pos, approach_distance=0.15, approach_axis='z')
        if done:
            print("[Task] Task ended after picking tomato2!")
            return

        # Step 9: place tomato2 on plate
        obs, reward, done = place(env, task, target_pos=plate_pos, approach_distance=0.15, approach_axis='z')
        if done:
            print("[Task] Task ended after placing tomato2!")
            return

        print("[Task] Task completed successfully! Reward:", reward)

    except Exception as e:
        print(f"[Task] Exception occurred: {e}")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()