# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use provided skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # positions: dict mapping object names to their 3D positions
        positions = get_object_positions()

        # === Plan Execution ===
        # Step 1: rotate (rotate gripper to 90 degrees for drawer manipulation)
        try:
            print("[Plan] Step 1: Rotate gripper to 90 degrees (drawer manipulation ready)")
            # Assume 'zero_deg' and 'ninety_deg' are known orientations; get current and target quaternion
            # If object_positions provides orientation, use it; else, use a hardcoded quaternion for 90deg about z
            # We'll use the current gripper pose as start, and rotate about z by 90deg
            obs = task.get_observation()
            start_quat = obs.gripper_pose[3:7]
            # 90 deg about z axis in quaternion (xyzw): [0, 0, sqrt(0.5), sqrt(0.5)]
            ninety_deg_quat = np.array([0, 0, np.sqrt(0.5), np.sqrt(0.5)])
            obs, reward, done = rotate(env, task, target_quat=ninety_deg_quat)
            if done:
                print("[Task] Task ended after rotate!")
                return
        except Exception as e:
            print(f"[Error] Step 1 (rotate) failed: {e}")
            shutdown_environment(env)
            return

        # Step 2: move (move gripper to side position of bottom drawer)
        try:
            print("[Plan] Step 2: Move gripper to bottom_side_pos (side of bottom drawer)")
            target_pos = positions.get('bottom_side_pos', None)
            if target_pos is None:
                raise KeyError("bottom_side_pos not found in object positions.")
            obs, reward, done = move(env, task, target_pos=target_pos)
            if done:
                print("[Task] Task ended after move to bottom_side_pos!")
                return
        except Exception as e:
            print(f"[Error] Step 2 (move to bottom_side_pos) failed: {e}")
            shutdown_environment(env)
            return

        # Step 3: move (move gripper to anchor position of bottom drawer)
        try:
            print("[Plan] Step 3: Move gripper to bottom_anchor_pos (anchor of bottom drawer)")
            target_pos = positions.get('bottom_anchor_pos', None)
            if target_pos is None:
                raise KeyError("bottom_anchor_pos not found in object positions.")
            obs, reward, done = move(env, task, target_pos=target_pos)
            if done:
                print("[Task] Task ended after move to bottom_anchor_pos!")
                return
        except Exception as e:
            print(f"[Error] Step 3 (move to bottom_anchor_pos) failed: {e}")
            shutdown_environment(env)
            return

        # Step 4: pick (grasp the drawer handle at anchor position)
        try:
            print("[Plan] Step 4: Pick (grasp) the bottom drawer handle at anchor position")
            # For picking the drawer, use the anchor position as the target
            target_pos = positions.get('bottom_anchor_pos', None)
            if target_pos is None:
                raise KeyError("bottom_anchor_pos not found in object positions.")
            obs, reward, done = pick(env, task, target_pos=target_pos, approach_distance=0.10, approach_axis='z')
            if done:
                print("[Task] Task ended after pick (drawer handle)!")
                return
        except Exception as e:
            print(f"[Error] Step 4 (pick drawer handle) failed: {e}")
            shutdown_environment(env)
            return

        # Step 5: pull (pull the drawer open)
        try:
            print("[Plan] Step 5: Pull the bottom drawer open")
            # Pull along x axis (assume drawer opens along +x), distance empirically set (e.g., 0.15m)
            pull_distance = 0.15
            obs, reward, done = pull(env, task, pull_distance=pull_distance, pull_axis='x')
            if done:
                print("[Task] Task ended after pull (open drawer)!")
                return
        except Exception as e:
            print(f"[Error] Step 5 (pull drawer) failed: {e}")
            shutdown_environment(env)
            return

        # Step 6: pick tomato1 from table
        try:
            print("[Plan] Step 6: Pick tomato1 from table")
            tomato1_pos = positions.get('tomato1', None)
            if tomato1_pos is None:
                raise KeyError("tomato1 not found in object positions.")
            obs, reward, done = pick(env, task, target_pos=tomato1_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended after pick (tomato1)!")
                return
        except Exception as e:
            print(f"[Error] Step 6 (pick tomato1) failed: {e}")
            shutdown_environment(env)
            return

        # Step 7: place tomato1 on plate
        try:
            print("[Plan] Step 7: Place tomato1 on plate")
            plate_pos = positions.get('plate', None)
            if plate_pos is None:
                raise KeyError("plate not found in object positions.")
            obs, reward, done = place(env, task, target_pos=plate_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended after place (tomato1)!")
                return
        except Exception as e:
            print(f"[Error] Step 7 (place tomato1) failed: {e}")
            shutdown_environment(env)
            return

        # Step 8: pick tomato2 from table
        try:
            print("[Plan] Step 8: Pick tomato2 from table")
            tomato2_pos = positions.get('tomato2', None)
            if tomato2_pos is None:
                raise KeyError("tomato2 not found in object positions.")
            obs, reward, done = pick(env, task, target_pos=tomato2_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended after pick (tomato2)!")
                return
        except Exception as e:
            print(f"[Error] Step 8 (pick tomato2) failed: {e}")
            shutdown_environment(env)
            return

        # Step 9: place tomato2 on plate
        try:
            print("[Plan] Step 9: Place tomato2 on plate")
            plate_pos = positions.get('plate', None)
            if plate_pos is None:
                raise KeyError("plate not found in object positions.")
            obs, reward, done = place(env, task, target_pos=plate_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended after place (tomato2)!")
                return
        except Exception as e:
            print(f"[Error] Step 9 (place tomato2) failed: {e}")
            shutdown_environment(env)
            return

        print("[Task] All plan steps executed successfully. Task complete.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
