import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use provided skills only

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # positions: dict mapping object names to their 3D positions
        positions = get_object_positions()

        # --- Step 1: Rotate gripper to ninety_deg ---
        # We assume the gripper's orientation for 'ninety_deg' is known or can be constructed.
        # If not, we use the current position and set the orientation accordingly.
        # Let's try to get the quaternion for 90 degrees rotation about z axis.
        from scipy.spatial.transform import Rotation as R
        obs = task.get_observation()
        gripper_pos = obs.gripper_pose[:3]
        # We'll rotate about z axis by 90 degrees (in radians)
        ninety_deg_quat = R.from_euler('z', 90, degrees=True).as_quat()  # xyzw
        # RLBench/pyrep expects [x, y, z, w] order
        # If the current orientation is not identity, we may need to compose rotations
        # For now, we use this as the target orientation
        try:
            obs, reward, done = rotate(env, task, target_quat=ninety_deg_quat)
            if done:
                print("[Task] Task ended after rotate!")
                return
        except Exception as e:
            print(f"[ERROR] Exception during rotate: {e}")
            shutdown_environment(env)
            return

        # --- Step 2: Move gripper to side-pos-bottom ---
        # Use the position of 'bottom_side_pos' from object_positions
        try:
            side_pos = positions.get('bottom_side_pos', None)
            if side_pos is None:
                side_pos = positions.get('side-pos-bottom', None)
            if side_pos is None:
                raise KeyError("bottom_side_pos or side-pos-bottom not found in object positions.")
            obs, reward, done = move(env, task, target_pos=np.array(side_pos))
            if done:
                print("[Task] Task ended after move to side-pos-bottom!")
                return
        except Exception as e:
            print(f"[ERROR] Exception during move to side-pos-bottom: {e}")
            shutdown_environment(env)
            return

        # --- Step 3: Move gripper to anchor-pos-bottom ---
        try:
            anchor_pos = positions.get('bottom_anchor_pos', None)
            if anchor_pos is None:
                anchor_pos = positions.get('anchor-pos-bottom', None)
            if anchor_pos is None:
                raise KeyError("bottom_anchor_pos or anchor-pos-bottom not found in object positions.")
            obs, reward, done = move(env, task, target_pos=np.array(anchor_pos))
            if done:
                print("[Task] Task ended after move to anchor-pos-bottom!")
                return
        except Exception as e:
            print(f"[ERROR] Exception during move to anchor-pos-bottom: {e}")
            shutdown_environment(env)
            return

        # --- Step 4: Pick the bottom drawer handle (simulate pick-drawer) ---
        # We use the pick() skill at the anchor position of the bottom drawer.
        try:
            obs, reward, done = pick(env, task, target_pos=np.array(anchor_pos), approach_distance=0.10, approach_axis='z')
            if done:
                print("[Task] Task ended after pick-drawer!")
                return
        except Exception as e:
            print(f"[ERROR] Exception during pick-drawer: {e}")
            shutdown_environment(env)
            return

        # --- Step 5: Pull the drawer open ---
        # We assume pulling along the x-axis is correct (drawer slides along x).
        # The pull distance may need to be tuned; let's use 0.15 meters as a reasonable value.
        try:
            obs, reward, done = pull(env, task, pull_distance=0.15, pull_axis='x')
            if done:
                print("[Task] Task ended after pull!")
                return
        except Exception as e:
            print(f"[ERROR] Exception during pull: {e}")
            shutdown_environment(env)
            return

        # --- Step 6: Pick up tomato1 from the table ---
        try:
            tomato1_pos = positions.get('tomato1', None)
            if tomato1_pos is None:
                raise KeyError("tomato1 not found in object positions.")
            obs, reward, done = pick(env, task, target_pos=np.array(tomato1_pos), approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended after pick tomato1!")
                return
        except Exception as e:
            print(f"[ERROR] Exception during pick tomato1: {e}")
            shutdown_environment(env)
            return

        # --- Step 7: Place tomato1 onto the plate ---
        try:
            plate_pos = positions.get('plate', None)
            if plate_pos is None:
                raise KeyError("plate not found in object positions.")
            obs, reward, done = place(env, task, target_pos=np.array(plate_pos), approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended after place tomato1!")
                return
        except Exception as e:
            print(f"[ERROR] Exception during place tomato1: {e}")
            shutdown_environment(env)
            return

        # --- Step 8: Pick up tomato2 from the table ---
        try:
            tomato2_pos = positions.get('tomato2', None)
            if tomato2_pos is None:
                raise KeyError("tomato2 not found in object positions.")
            obs, reward, done = pick(env, task, target_pos=np.array(tomato2_pos), approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended after pick tomato2!")
                return
        except Exception as e:
            print(f"[ERROR] Exception during pick tomato2: {e}")
            shutdown_environment(env)
            return

        # --- Step 9: Place tomato2 onto the plate ---
        try:
            obs, reward, done = place(env, task, target_pos=np.array(plate_pos), approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended after place tomato2! Reward:", reward)
                return
        except Exception as e:
            print(f"[ERROR] Exception during place tomato2: {e}")
            shutdown_environment(env)
            return

        print("[Task] Task completed successfully! Reward:", reward)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()