import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use provided skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Task: Open one of the drawers, then put all tomatoes on the plate.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Object names from object list and get_object_positions:
        # 'bottom_anchor_pos', 'bottom_joint_pos', 'bottom_side_pos',
        # 'middle_anchor_pos', 'middle_joint_pos', 'middle_side_pos',
        # 'top_anchor_pos', 'top_joint_pos', 'top_side_pos',
        # 'waypoint1', 'rubbish', 'tomato1', 'tomato2', 'plate'

        # We'll use the bottom drawer for simplicity
        drawer_side_pos = positions.get('bottom_side_pos', None)
        drawer_anchor_pos = positions.get('bottom_anchor_pos', None)
        drawer_joint_pos = positions.get('bottom_joint_pos', None)
        tomato1_pos = positions.get('tomato1', None)
        tomato2_pos = positions.get('tomato2', None)
        plate_pos = positions.get('plate', None)

        # Safety check for required positions
        required_keys = ['bottom_side_pos', 'bottom_anchor_pos', 'bottom_joint_pos', 'tomato1', 'tomato2', 'plate']
        for key in required_keys:
            if positions.get(key, None) is None:
                print(f"[Error] Required object position '{key}' not found in get_object_positions().")
                shutdown_environment(env)
                return

        # === PLAN EXECUTION ===
        # Step 1: rotate (rotate gripper to 90 degrees for side approach)
        # We'll use a quaternion for 90 deg rotation about z axis
        # RLBench uses [x, y, z, w] format
        from scipy.spatial.transform import Rotation as R
        ninety_deg_quat = R.from_euler('z', 90, degrees=True).as_quat()  # [x, y, z, w]
        try:
            obs, reward, done = rotate(env, task, target_quat=ninety_deg_quat)
            if done:
                print("[Task] Task ended after rotate!")
                return
        except Exception as e:
            print(f"[Error] Exception during rotate: {e}")
            shutdown_environment(env)
            return

        # Step 2: move (move to side of bottom drawer)
        try:
            obs, reward, done = move(env, task, target_pos=drawer_side_pos)
            if done:
                print("[Task] Task ended after move to drawer side!")
                return
        except Exception as e:
            print(f"[Error] Exception during move to drawer side: {e}")
            shutdown_environment(env)
            return

        # Step 3: move (move to anchor position of bottom drawer)
        try:
            obs, reward, done = move(env, task, target_pos=drawer_anchor_pos)
            if done:
                print("[Task] Task ended after move to drawer anchor!")
                return
        except Exception as e:
            print(f"[Error] Exception during move to drawer anchor: {e}")
            shutdown_environment(env)
            return

        # Step 4: pick (grasp the drawer handle at anchor position)
        # Use pick with approach from above (z axis)
        try:
            obs, reward, done = pick(env, task, target_pos=drawer_anchor_pos, approach_distance=0.10, approach_axis='z')
            if done:
                print("[Task] Task ended after picking drawer handle!")
                return
        except Exception as e:
            print(f"[Error] Exception during pick drawer handle: {e}")
            shutdown_environment(env)
            return

        # Step 5: pull (pull the drawer open along x axis, e.g., 0.15m)
        try:
            obs, reward, done = pull(env, task, pull_distance=0.15, pull_axis='x')
            if done:
                print("[Task] Task ended after pulling drawer!")
                return
        except Exception as e:
            print(f"[Error] Exception during pull: {e}")
            shutdown_environment(env)
            return

        # Step 6: pick (pick up tomato1 from table)
        try:
            obs, reward, done = pick(env, task, target_pos=tomato1_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended after picking tomato1!")
                return
        except Exception as e:
            print(f"[Error] Exception during pick tomato1: {e}")
            shutdown_environment(env)
            return

        # Step 7: place (place tomato1 on plate)
        try:
            obs, reward, done = place(env, task, target_pos=plate_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended after placing tomato1!")
                return
        except Exception as e:
            print(f"[Error] Exception during place tomato1: {e}")
            shutdown_environment(env)
            return

        # Step 8: pick (pick up tomato2 from table)
        try:
            obs, reward, done = pick(env, task, target_pos=tomato2_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended after picking tomato2!")
                return
        except Exception as e:
            print(f"[Error] Exception during pick tomato2: {e}")
            shutdown_environment(env)
            return

        # Step 9: place (place tomato2 on plate)
        try:
            obs, reward, done = place(env, task, target_pos=plate_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended after placing tomato2!")
                return
        except Exception as e:
            print(f"[Error] Exception during place tomato2: {e}")
            shutdown_environment(env)
            return

        print("[Task] Task completed successfully! All tomatoes placed on plate after opening drawer.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()