import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # --- Object name mapping (from object list and environment) ---
        # Drawer positions (anchor/side/joint) and plate
        # The object list includes:
        # 'bottom_anchor_pos', 'bottom_joint_pos', 'bottom_side_pos',
        # 'middle_anchor_pos', 'middle_joint_pos', 'middle_side_pos',
        # 'top_anchor_pos', 'top_joint_pos', 'top_side_pos',
        # 'waypoint1', 'rubbish', 'tomato1', 'tomato2', 'plate'
        #
        # We'll use the bottom drawer for the "pull open any drawer" step.
        # We'll use 'tomato1', 'tomato2', and 'plate' for the tomato placement.

        # Defensive: check all required objects exist
        required_keys = [
            'bottom_side_pos', 'bottom_anchor_pos', 'bottom_joint_pos',
            'tomato1', 'tomato2', 'plate'
        ]
        for k in required_keys:
            if k not in positions:
                raise RuntimeError(f"Required object '{k}' not found in object_positions.")

        # Step 1: rotate (rotate gripper to 90 degrees for side approach)
        # We'll use quaternion for 90 deg rotation about z axis
        # Assume initial orientation is zero_deg (identity quaternion)
        # RLBench uses [x, y, z, w] order for quaternions
        from scipy.spatial.transform import Rotation as R
        # 90 deg about z axis
        ninety_deg_quat = R.from_euler('z', 90, degrees=True).as_quat()  # [x, y, z, w]
        try:
            obs, reward, done = rotate(
                env,
                task,
                target_quat=ninety_deg_quat,
                max_steps=100,
                threshold=0.05,
                timeout=10.0
            )
            if done:
                print("[Task] Task ended after rotate!")
                return
        except Exception as e:
            print(f"[Task] Exception during rotate: {e}")
            shutdown_environment(env)
            raise

        # Step 2: move to side of bottom drawer
        try:
            bottom_side_pos = positions['bottom_side_pos']
            obs, reward, done = move(
                env,
                task,
                target_pos=bottom_side_pos,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            if done:
                print("[Task] Task ended after move to bottom_side_pos!")
                return
        except Exception as e:
            print(f"[Task] Exception during move to bottom_side_pos: {e}")
            shutdown_environment(env)
            raise

        # Step 3: move to anchor of bottom drawer (for grasping handle)
        try:
            bottom_anchor_pos = positions['bottom_anchor_pos']
            obs, reward, done = move(
                env,
                task,
                target_pos=bottom_anchor_pos,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            if done:
                print("[Task] Task ended after move to bottom_anchor_pos!")
                return
        except Exception as e:
            print(f"[Task] Exception during move to bottom_anchor_pos: {e}")
            shutdown_environment(env)
            raise

        # Step 4: move to joint of bottom drawer (simulate grasping the handle)
        try:
            bottom_joint_pos = positions['bottom_joint_pos']
            obs, reward, done = move(
                env,
                task,
                target_pos=bottom_joint_pos,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            if done:
                print("[Task] Task ended after move to bottom_joint_pos!")
                return
        except Exception as e:
            print(f"[Task] Exception during move to bottom_joint_pos: {e}")
            shutdown_environment(env)
            raise

        # Step 5: pull (pull the drawer open along x axis, e.g., 0.15m)
        try:
            pull_distance = 0.15  # meters, adjust as needed for your environment
            obs, reward, done = pull(
                env,
                task,
                pull_distance=pull_distance,
                pull_axis='x',
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            if done:
                print("[Task] Task ended after pull!")
                return
        except Exception as e:
            print(f"[Task] Exception during pull: {e}")
            shutdown_environment(env)
            raise

        # Step 6: pick tomato1 from table
        try:
            tomato1_pos = positions['tomato1']
            obs, reward, done = pick(
                env,
                task,
                target_pos=tomato1_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Task] Task ended after pick tomato1!")
                return
        except Exception as e:
            print(f"[Task] Exception during pick tomato1: {e}")
            shutdown_environment(env)
            raise

        # Step 7: place tomato1 on plate
        try:
            plate_pos = positions['plate']
            obs, reward, done = place(
                env,
                task,
                target_pos=plate_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Task] Task ended after place tomato1!")
                return
        except Exception as e:
            print(f"[Task] Exception during place tomato1: {e}")
            shutdown_environment(env)
            raise

        # Step 8: pick tomato2 from table
        try:
            tomato2_pos = positions['tomato2']
            obs, reward, done = pick(
                env,
                task,
                target_pos=tomato2_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Task] Task ended after pick tomato2!")
                return
        except Exception as e:
            print(f"[Task] Exception during pick tomato2: {e}")
            shutdown_environment(env)
            raise

        # Step 9: place tomato2 on plate
        try:
            obs, reward, done = place(
                env,
                task,
                target_pos=plate_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Task] Task ended after place tomato2!")
                return
        except Exception as e:
            print(f"[Task] Exception during place tomato2: {e}")
            shutdown_environment(env)
            raise

        print("[Task] Task completed successfully! Reward:", reward)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()